package com.fshows.com.fbank.openapi.sdk.service;

import com.alibaba.fastjson.JSONObject;
import com.fshows.com.fbank.openapi.sdk.client.OpenParameters;
import com.fshows.com.fbank.openapi.sdk.config.Configuration;
import com.fshows.com.fbank.openapi.sdk.config.FileConfiguration;
import com.fshows.com.fbank.openapi.sdk.constant.FileDownloadStatusEnum;
import com.fshows.com.fbank.openapi.sdk.constant.FileResponseConstants;
import com.fshows.com.fbank.openapi.sdk.constant.RequestConstants;
import com.fshows.com.fbank.openapi.sdk.crypto.CryptoSignatureService;
import com.fshows.com.fbank.openapi.sdk.util.CryptoServiceUtils;
import org.apache.commons.codec.digest.DigestUtils;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ThreadLocalRandom;

/**
 * 文件网关文件下载服务
 *
 * @author miwenming
 * @date 2020/5/25 15:59
 */
public class OpenApiFileDownloadService extends AbstractOpenApiService {

    @Override
    protected String getRemoteAddress(Configuration configuration) {
        String remoteFileAddress = configuration.remoteAddress();
        if (configuration instanceof FileConfiguration) {
            remoteFileAddress = ((FileConfiguration) configuration).remoteFileAddress();
        }

        if (!remoteFileAddress.endsWith("/")) {
            remoteFileAddress = remoteFileAddress + "/";
        }

        String path = getFilePath();
        if (path.startsWith("/")) {
            path = path.substring(1);
        }

        return remoteFileAddress + path;
    }

    protected String getFilePath() {
        return "api/download";
    }

    @Override
    protected void setAlgorithm(Configuration configuration, HttpURLConnection connection, String moduleCode) {
        connection.setRequestProperty(RequestConstants.RequestHeader.KEY_ALGORITHM,
                RequestConstants.AlgorithmService.ALGORITHM_TYPE_RSA_FILE);
    }

    @Override
    public String verifyAndDecrypt(HttpURLConnection connection, Configuration configuration,
                                   OpenParameters context, byte[] response) throws Exception {
        String bizContent = getBizContent(connection);

        verifyFile(configuration, bizContent, response);

        downloadFile(configuration, context, response, bizContent);
        return "file download success";
    }

    /**
     * 获取请求头中的响应业务报文
     *
     * @param connection HttpURLConnection对象
     */
    private String getBizContent(HttpURLConnection connection) {
        String bizContent = connection.getHeaderField(FileResponseConstants.ResponseHeader.KEY_BIZ_CONTENT);
        String bizContentDecode;
        try {
            bizContentDecode = URLDecoder.decode(bizContent, StandardCharsets.UTF_8.name());
        } catch (UnsupportedEncodingException e) {
            bizContentDecode = bizContent;
        }

        logger.info("bizContent is {}", bizContentDecode);
        return bizContentDecode;
    }

    /**
     * 校验文件请求响应状态。
     * 如后续状态更新，规则更改，可以重写改方法
     *
     * @param signDataJO 签名数据
     */
    protected static void checkFileStatus(JSONObject signDataJO) {
        String status = signDataJO.getString(FileResponseConstants.SignData.KEY_STATUS);
        if (FileDownloadStatusEnum.SUCCESS.getStatus().equalsIgnoreCase(status)) {
            return;
        }

        for (FileDownloadStatusEnum fileDownloadStatus : FileDownloadStatusEnum.values()) {
            if (fileDownloadStatus.getStatus().equalsIgnoreCase(status)) {
                throw new RuntimeException(fileDownloadStatus.getMessage());
            }
        }

        throw new RuntimeException(FileDownloadStatusEnum.OTHER.getMessage());
    }

    /**
     * 校验文件请求响应码 非000000即失败
     *
     * @param bizContentJO 响应头中的业务报文
     */
    private void checkResponseCode(JSONObject bizContentJO) {
        String code = bizContentJO.getString(FileResponseConstants.BizContent.KEY_CODE);
        if (!FileResponseConstants.SUCCESS_CODE.equalsIgnoreCase(code)) {
            throw new RuntimeException(bizContentJO.getString(FileResponseConstants.BizContent.KEY_MESSAGE));
        }
    }

    /**
     * 验签
     *
     * @param configuration 配置类
     * @param bizContent    业务报文
     * @param response      响应文件流转换后的字符串
     */
    private void verifyFile(Configuration configuration, String bizContent, byte[] response) throws Exception {
        JSONObject bizContentJO = JSONObject.parseObject(bizContent);
        checkResponseCode(bizContentJO);

        JSONObject signDataJO = bizContentJO.getJSONObject(FileResponseConstants.BizContent.KEY_SIGN_DATA);
        verifySignature(configuration, bizContentJO, signDataJO);
        checkFileStatus(signDataJO);
        checkFileMD5(signDataJO, response);
    }

    /**
     * 验签
     *
     * @param configuration 配置类
     * @param bizContentJO 业务报文json对象
     * @param signDataJO 签名报文json对象
     */
    private void verifySignature(Configuration configuration, JSONObject bizContentJO, JSONObject signDataJO)
            throws Exception {
        String signature = bizContentJO.getString(FileResponseConstants.BizContent.KEY_SIGNATURE);
        CryptoSignatureService cryptoSignatureService
                = CryptoServiceUtils.getCryptoSignatureService(configuration.signType());

        Map<String, Object> signDataTreeMap = new TreeMap<>(signDataJO);
        JSONObject signDataSorted = new JSONObject(signDataTreeMap);
        boolean verifyResult = cryptoSignatureService.verifySignature(
                signDataSorted.toJSONString(), signature, configuration.fbankSignPubKey());
        if (!verifyResult) {
            throw new RuntimeException("verifyFile failed!");
        }
    }

    /**
     * 比对文件MD5
     *
     * @param signDataJO 签名数据
     * @param response 文件流字符串
     */
    private void checkFileMD5(JSONObject signDataJO, byte[] response) {
        String responseMd5 = DigestUtils.md5Hex(response);
        String signDataMd5 = signDataJO.getString(FileResponseConstants.SignData.KEY_MD5);
        logger.info("request md5 is {}, get md5 from file is {}", signDataMd5, responseMd5);
        if (!responseMd5.equalsIgnoreCase(signDataMd5)) {
            throw new RuntimeException("md5 check failed");
        }
    }

    private void downloadFile(Configuration configuration,
                              OpenParameters context, byte[] response, String bizContent) {
        Path path = Paths.get(getFileName(configuration, context, bizContent));
        try {
            if (Files.notExists(path)) {
                Files.createFile(path);
            } else {
                throw new RuntimeException("file already exists");
            }
            Files.write(path, response);
        } catch (IOException e) {
            throw new RuntimeException("file download failed");
        }
    }

    /**
     * 获取文件名称
     *
     * @param configuration 配置类
     * @param context 上下文
     * @param bizContent 返回的报文头里的业务数据
     * @return
     */
    private String getFileName(Configuration configuration, OpenParameters context, String bizContent) {
        JSONObject bizContentJO = JSONObject.parseObject(bizContent);
        JSONObject signData = bizContentJO.getJSONObject(FileResponseConstants.BizContent.KEY_SIGN_DATA);
        String fileType = signData.getString(FileResponseConstants.SignData.KEY_FILE_TYPE);
        FileConfiguration fileConfiguration = (FileConfiguration) configuration;
        String storagePath = fileConfiguration.storagePath();
        if (!storagePath.endsWith("/")) {
            storagePath = storagePath + "/";
        }

        String fileName = context.getFileName();
        if (fileName == null) {
            fileName = String.valueOf(ThreadLocalRandom.current().nextInt());
        }
        if (!fileName.contains(".")) {
            fileName = fileName + "." + fileType;
        }

        return storagePath + fileName;
    }

}
