/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.easypay.sdk.client.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.PropertyNamingStrategy;
import com.alibaba.fastjson.serializer.SerializeConfig;
import com.fshows.easypay.sdk.client.EasyPayMerchantClient;
import com.fshows.easypay.sdk.constant.CommonConstant;
import com.fshows.easypay.sdk.enums.EasyMerchantApiEnum;
import com.fshows.easypay.sdk.exception.EasyPayException;
import com.fshows.easypay.sdk.request.base.EasyPayMerchantBaseRequest;
import com.fshows.easypay.sdk.response.base.EasyPayMerchantBizResponse;
import com.fshows.easypay.sdk.util.FsHttpUtil;
import com.fshows.easypay.sdk.util.SignUtil;
import com.fshows.easypay.sdk.util.ValidateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;

import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

/**
 * @author zhaoxumin
 * @version EasyPayBaseClientImpl.java, v 0.1 2023-10-27 13:32
 */
@Slf4j
public class EasyPayMerchantClientImpl implements EasyPayMerchantClient {

    /**
     * 应用appId
     */
    private final String appId;

    private final String url;
    /**
     * 签名类型
     */
    private final String signType;
    /**
     * 密钥
     */
    private final String key;

    /**
     * http连接超时时间
     */
    protected int connectionTimeout = CommonConstant.DEFAULT_CONNECTION_TIMEOUT;

    /**
     * http response读取超时时间
     */
    protected int readTimeout = CommonConstant.DEFAULT_READ_TIMEOUT;


    private static final SerializeConfig SNAKE_CASE_CONFIG = new SerializeConfig();

    static {
        SNAKE_CASE_CONFIG.propertyNamingStrategy = PropertyNamingStrategy.SnakeCase;
    }


    public EasyPayMerchantClientImpl(String appId, String signType, String url, String key) {
        this.appId = appId;
        this.signType = signType;
        this.url = url;
        this.key = key;
    }

    /**
     * 执行
     * 请求易生接口
     *
     * @param request              请求参数
     * @param easyPayApiDefinition 方法枚举
     * @return UmBaseResponse
     * <p>
     * EasyPayApiDefinition api方法请参考下列API枚举
     * @throws EasyPayException 轻松支付例外
     */
    @Override
    public <R extends EasyPayMerchantBizResponse, T> EasyPayMerchantBizResponse<R> execute(EasyPayMerchantBaseRequest<R> request, EasyPayApiDefinition easyPayApiDefinition) throws EasyPayException {
        return execute(request, easyPayApiDefinition, this.appId);
    }

    /**
     * 执行
     * 请求易生接口
     *
     * @param request    请求参数
     * @param definition 方法枚举
     * @param subAppId   子应用id
     * @return UmBaseResponse
     * @throws EasyPayException 轻松支付例外
     */
    @Override
    public <R extends EasyPayMerchantBizResponse, T> EasyPayMerchantBizResponse<R> execute(EasyPayMerchantBaseRequest<R> request, EasyPayApiDefinition definition, String subAppId) throws EasyPayException {
        String method = definition.getMethod();
        //验证参数
        validateParam(request, method);
        //获取开始时间
        final long startTime = System.currentTimeMillis();
        String url = this.url + method;
        try {
            //获取请求参数
            buildEasyPayBizRequest(request, StringUtils.defaultIfBlank(subAppId, this.appId));
            final Map<String, String> requestData = getRequestData(request, StringUtils.defaultIfBlank(subAppId, this.appId));
            log.info("【easypay-sdk】接口调用开始 >> url={}, request={}, requestData={}", url, JSON.toJSONString(request), requestData);
            Map<String, String> headers = new HashMap<>();
            headers.put("Accept", CommonConstant.ACCEPT_TYPE);
            String easyPayResponse = FsHttpUtil.post(url, requestData, CommonConstant.UTF_8, CommonConstant.JSON_CONTENT_TYPE_URLENCODED, connectionTimeout, readTimeout, headers);
            log.info("record bad-liquidation info >> success >> 【easypay-sdk】接口调用结束 >> url={}, request={},cost={}", url, request, System.currentTimeMillis() - startTime);
            log.info("【easypay-sdk】接口调用结束 >> url={}, request={}, requestData={}, response={}, cost={}", url, JSON.toJSONString(request), requestData, easyPayResponse, System.currentTimeMillis() - startTime);
            return parseResponse(easyPayResponse, request);
        } catch (Exception ex) {
            log.error("record bad-liquidation info >> fail >>【easypay-sdk】接口调用失败 >> url={}, request={},ex={},cost={}", url, request, ExceptionUtils.getStackTrace(ex), System.currentTimeMillis() - startTime);
            log.error("【easypay-sdk】接口调用失败 >> url={}, request={},ex={},cost={}", url, request, ExceptionUtils.getStackTrace(ex), System.currentTimeMillis() - startTime);
            throw new EasyPayException(ex.getMessage());
        }
    }

    private <R extends EasyPayMerchantBizResponse> void buildEasyPayBizRequest(EasyPayMerchantBaseRequest<R> request, String appId) {
        request.setClientCode(appId);
        request.setVersion("1.0");
    }

    /**
     * 结果解析
     *
     * @param result  返回结果
     * @param request 请求参数
     * @param <R>     结果
     * @return UmBaseResponse
     */
    @SuppressWarnings("unchecked")
    private <R extends EasyPayMerchantBizResponse, T> EasyPayMerchantBizResponse<R> parseResponse(String result, EasyPayMerchantBaseRequest<R> request) {
        EasyPayMerchantBizResponse bizResponse = JSON.parseObject(result, EasyPayMerchantBizResponse.class);
        bizResponse.setData(JSON.parseObject(result, request.getResponseClass()));
        return bizResponse;
    }

    /**
     * 获取签名参数
     *
     * @param request 请求参数
     * @param <R>     返参
     * @return Map<String, String>
     */
    private <R extends EasyPayMerchantBizResponse> Map<String, String> getRequestData(EasyPayMerchantBaseRequest<R> request, String appId) throws EasyPayException {
        TreeMap<String, String> parameterMap = SignUtil.getMerchantSignParameterMap(request);
        parameterMap.put("clientCode", appId);
        parameterMap.put("version", "1.0");
        String sign = SignUtil.generateMd5Sign(parameterMap, key);
        request.setMAC(sign);
        parameterMap.put("MAC", sign);
        return parameterMap;
    }

    /**
     * 参数校验
     *
     * @param request 请求参数
     * @param method  请求方法
     * @param <R>
     */
    private <R extends EasyPayMerchantBizResponse> void validateParam(EasyPayMerchantBaseRequest<R> request, String method) {
        if (request == null) {
            throw new IllegalArgumentException("接口请求参数不能为空");
        }
        if (StringUtils.isBlank(key)) {
            throw new IllegalArgumentException("签名密钥不能为空");
        }
        if (StringUtils.isBlank(signType)) {
            throw new IllegalArgumentException("签名类型不能为空");
        }
        if (StringUtils.isBlank(method)) {
            throw new IllegalArgumentException("请求方法不能为空");
        }
        if (null == EasyMerchantApiEnum.getByValue(method)) {
            throw new IllegalArgumentException("请求方法不存在");
        }
        //注解验证
        ValidateUtil.validateWithThrow(request);
    }
}