/**
 * fshows.com
 * Copyright (C) 2013-2023 All Rights Reserved.
 */
package com.fshows.easypay.sdk.client.impl;

import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSONObject;
import com.fshows.easypay.sdk.client.EasyPayUploadClient;
import com.fshows.easypay.sdk.exception.EasyPayException;
import com.fshows.easypay.sdk.request.merchant.UploadPicRequest;
import com.fshows.easypay.sdk.response.merchant.UploadPicResponse;
import com.fshows.easypay.sdk.util.SignUtil;
import com.fshows.easypay.sdk.util.ValidateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.FileBody;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;

import javax.validation.constraints.NotBlank;
import java.io.File;
import java.io.IOException;
import java.util.Map;
import java.util.TreeMap;

/**
 * @author zhubo
 * @version EasyPayUploadClientImpl.java, v 0.1 2023-11-04 16:00 zhubo
 */
@Slf4j
public class EasyPayUploadClientImpl implements EasyPayUploadClient {
    /**
     * 机构号
     */
    @NotBlank
    private final String orgId;
    /**
     * 签名类型
     */
    @NotBlank
    private final String signType;
    /**
     * 易生接口网关根URL
     */
    @NotBlank
    protected String postUrl;
    /**
     * 用于加签的付呗私钥
     */
    @NotBlank
    private String fubeiPrivateKey;


    public EasyPayUploadClientImpl(String postUrl, String orgId, String signType, String fubeiPrivateKey) {
        this.postUrl = postUrl;
        this.orgId = orgId;
        this.signType = signType;
        this.fubeiPrivateKey = fubeiPrivateKey;
    }


    @Override
    public UploadPicResponse upload(UploadPicRequest request) throws EasyPayException {
        return upload(request, null);
    }

    @Override
    public UploadPicResponse upload(UploadPicRequest request, String subAgentId) throws EasyPayException {
        ValidateUtil.validateWithThrow(request);
        File file = request.getFileName();
        CloseableHttpClient httpclient = HttpClients.createDefault();
        String responseString = "";
        try {
            HttpPost httpPost = new HttpPost(this.postUrl);
            MultipartEntityBuilder entity = MultipartEntityBuilder.create();
            Map<String, String> parameters = getRequestParameters(request, subAgentId);
            // 请求参数
            for (String key : parameters.keySet()) {
                entity.addTextBody(key, parameters.get(key), ContentType.DEFAULT_TEXT);
            }
            entity.addPart("fileName", new FileBody(file));
            httpPost.setEntity(entity.build());
            RequestConfig requestConfig = RequestConfig.custom().setSocketTimeout(50000).setConnectTimeout(3000).build();
            httpPost.setConfig(requestConfig);
            log.info("【easypay-sdk】请求接口开始 >> 图片上传开始 >> url={}, param={}, request={} ", this.postUrl, parameters, httpPost.getRequestLine());
            CloseableHttpResponse response = httpclient.execute(httpPost);
            if (response.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
                HttpEntity httpEntity = response.getEntity();
                log.info("【easypay-sdk】请求接口结束 >> 图片上传出参 >> response = " + response.getStatusLine());
                try {
                    if (httpEntity != null) {
                        responseString = EntityUtils.toString(httpEntity);
                        log.info("【easypay-sdk】图片上传出参 >> response content:" + responseString);
                        EntityUtils.consume(httpEntity);
                    }
                } finally {
                    response.close();
                }
            }
        } catch (Exception e) {
            log.error("【easypay-sdk】请求异常 >> 图片上传异常", e);
            throw new EasyPayException("易生图片上传失败");
        } finally {
            if (httpclient != null) {
                try {
                    httpclient.close();
                } catch (IOException e) {
                    log.warn("资源关闭异常", e);
                    e.printStackTrace();
                }
            }
        }
        return StrUtil.isBlank(responseString) ? null : JSONObject.parseObject(responseString, UploadPicResponse.class);
    }

    /**
     * 获取签名参数
     *
     * @param request 请求参数
     * @param <R>     返参
     * @return Map<String, String>
     */
    private <R> Map<String, String> getRequestParameters(UploadPicRequest request, String appId) throws Exception {
        TreeMap<String, String> parameterMap = SignUtil.getParameterMap(request);
        parameterMap.put("clientCode", StrUtil.blankToDefault(appId, this.orgId));
        parameterMap.put("version", "1.0");
        parameterMap.remove("fileName");
        String sign = SignUtil.generateMd5Sign(parameterMap, fubeiPrivateKey);
        parameterMap.put("MAC", sign);
        return parameterMap;
    }
}
