/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.request;

import lombok.Data;
import lombok.ToString;
import org.hibernate.validator.constraints.Length;

import javax.validation.constraints.NotBlank;

/**
 * 统一下单 request
 *
 * @author
 * @version CmbcLcbpPayRequest.java, v 0.1 2025-02-10 15:37:48
 */
@Data
@ToString(callSuper = true)
public class CmbcLcbpPayRequest extends CmbcBizRequest {
    private static final long serialVersionUID = 136353599918433327L;

    /**
     * 支付类型
     * API_WXQRCODE 微信正扫
     * API_WXSCAN 微信反扫
     * H5_WXJSAPI 微信公众号跳转支付
     * API_ZFBQRCODE 支付宝正扫
     * API_ZFBSCAN 支付宝反扫
     * H5_ZFBJSAPI 支付宝服务窗
     * API_UNIONQRCODE 银联聚合码（原银联正扫）
     * API_CMBCSCAN 民生反扫 支持以上涉及的所有反扫类型
     * WX_APPLET 微信小程序
     *
     * (必填)
     */
    @NotBlank(message = "支付类型不能为空")
    @Length(max = 30, message = "支付类型长度不能超过30")
    private String selectTradeType;

    /**
     * 交易金额
     * 交易金额，以分为单位
     * (必填)
     */
    @NotBlank(message = "交易金额不能为空")
    @Length(max = 30, message = "交易金额长度不能超过30")
    private String amount;

    /**
     * 订单信息
     * 商户订单内容，商品信息，该字段不允许包含 '|,|\\|`（竖线为分隔符）
     * (必填)
     */
    @NotBlank(message = "订单信息不能为空")
    @Length(max = 128, message = "订单信息长度不能超过128")
    private String orderInfo;

    /**
     * 商户订单号
     * 商户订单号(须保证唯一)，建议是商户平台号+8 位日期+商户自定的订单号
     * (必填)
     */
    @NotBlank(message = "商户订单号不能为空")
    @Length(max = 48, message = "商户订单号长度不能超过48")
    private String merchantSeq;

    /**
     * 商户流水号
     * 商户流水号，本次交易流水，商户自主生成
     * (必填)
     */
    @NotBlank(message = "商户流水号不能为空")
    @Length(max = 32, message = "商户流水号长度不能超过32")
    private String mchSeqNo;

    /**
     * 订单日期
     * yyyyMMdd
     * (必填)
     */
    @NotBlank(message = "订单日期不能为空")
    @Length(max = 8, message = "订单日期长度不能超过8")
    private String transDate;

    /**
     * 订单时间
     * yyyyMMddHHmmssSSS
     * (必填)
     */
    @NotBlank(message = "订单时间不能为空")
    @Length(max = 17, message = "订单时间长度不能超过17")
    private String transTime;

    /**
     * 通知地址
     * 商户实现的接收异步通知的 url 地址，非必输字段，商户上送 url 会接收到通知，不上送则没有通知
     * (选填)
     */
    @Length(max = 512, message = "通知地址长度不能超过512")
    private String notifyUrl;

    /**
     * 备注信息
     * 反扫模式下该部分必输，填扫描客户二维码 Base64 后的值
     * (选填)
     */
    @Length(max = 200, message = "备注信息长度不能超过200")
    private String remark;

    /**
     * 公众号appId
     * 微信公众号支付|微信小程序 API 下，该部分必输，填子公众号的appId
     * (选填)
     */
    @Length(max = 18, message = "公众号 appId 长度不能超过18")
    private String subAppId;

    /**
     * 用户openId
     * 微信公众号支付|微信小程序 API 下，该部分必输，填用户的openId
     * (选填)
     */
    @Length(max = 28, message = "用户 openId 长度不能超过28")
    private String subOpenId;

    /**
     * 用户标识
     * 支付宝服务窗下，该字段必输
     * (选填)
     */
    @Length(max = 16, message = "用户标识长度不能超过16")
    private String userId;

    /**
     * 服务商附加信息
     * 服务商附言，长度限制 40 个字符，超长截断；当选择使用“数币扫聚合码”支付时，输入则展示在支付结果页面，该字段不允许包含|（竖线为分隔符）
     * (选填)
     */
    @Length(max = 40, message = "服务商附加信息长度不能超过40")
    private String fuyan;

    /**
     * 扩展字段
     * 该字段用于上送扩展字段，格式为： key1=value1|key2=value2
     * (选填)
     */
    private String reserve;

    /**
     * 设备号
     * (选填)
     */
    private String deviceInfo;

    /**
     * 扩展字段1
     * 上送透传给第三方的信息
     * (选填)
     */
    private String inExtData;

    /**
     * 定位信息
     * 受理终端设备实时经纬度信息，格式为纬度/经度，+表示北纬、东经， -表示南纬、西经
     * 示例：E116.397005,N39.956054
     * (选填)
     */
    private String gps;

    /**
     * 基站信息
     * 完整的格式为：mcc(国家码)-mnc（运营商）-lac（区域码）-cid（基站编号）-sim 卡号-基站信号
     * (选填)
     */
    private String station;

    /**
     * 商户端设备 IP
     * 商户端终端设备 IP 地址
     * (选填)
     */
    private String termIP;

    /**
     * 终端入网认证编
     * 银行卡受理终端产品入网认证编号
     * (选填)
     */
    private String netLicense;

    /**
     * 终端设备类型
     * 01：自动柜员机（含 ATM 和 CDM）和多媒体自助终端 02：传统 POS 03：mPOS 04：智能 POS 05：II 型固定电话 06：云闪付终端； 07：保留使用； 08：手机 POS； 09：刷脸付终端； 10：条码支付受理终端； 11：条码支付辅助受理终端
     * (选填)
     */
    private String termType;

    /**
     * 终端设备的硬件序列号
     * (选填)
     */
    private String serialNum;

    /**
     * 加密随机因子
     * 仅在被扫支付类交易报文中出现：若付款码为 19 位数字，则取后 6 位；若付款码码为 EMV 二维码，则取其 tag 57 的卡号/token 号的后 6 位
     * (选填)
     */
    private String randNum;

    /**
     * 加密密文
     * 仅在被扫支付类交易报文中出现：64bit 的密文数据，对终端硬件序列号和加密随机因子加密后的结果。本子域取值为：64bit 密文数据进行 base64 编码后的结果
     * (选填)
     */
    private String secrtText;

    /**
     * 终端应用程序的版本号
     * 终端应用程序的版本号
     * (选填)
     */
    private String appVersion;

    /**
     * 发起方式
     * 01=智能 POS 02=Q码盒子 03=手机/PAD 11=台牌 21=网络接口（API） 22=乐收银商超 PC/H5
     * (选填)
     */
    private String chnlType;

    /**
     * 银联二维码服务商 ID
     * (选填)
     */
    private String upayPID;
}