/**
 * fshows.com
 * Copyright (C) 2013-2023 All Rights Reserved.
 */
package
        com.fshows;

import cfca.internal.tool.ASN1Parser;
import cfca.org.bouncycastle.asn1.ASN1InputStream;
import cfca.sm.algorithm.SM2Pfx;
import cfca.sm2.signature.SM2PrivateKey;
import cfca.sm2rsa.common.Mechanism;
import cfca.sm2rsa.common.PKIException;
import cfca.util.Base64;
import cfca.util.CertUtil;
import cfca.util.EnvelopeUtil;
import cfca.util.KeyUtil;
import cfca.util.SignatureUtil2;
import cfca.util.cipher.lib.JCrypto;
import cfca.util.cipher.lib.Session;
import cfca.x509.certificate.X509Cert;
import cfca.x509.certificate.X509CertHelper;
import cn.hutool.core.bean.BeanUtil;
import com.alibaba.fastjson.JSONObject;
import com.fshows.apienum.CmbcTradeApiDefinitionEnum;
import com.fshows.request.CmbcBizRequest;
import com.fshows.response.CmbcBaseResponse;
import com.fshows.response.CmbcBizResponse;
import com.fshows.sdk.core.client.base.AbstractApiClient;
import com.fshows.sdk.core.client.base.handler.IHttpRequestHandler;
import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.ClientInfoModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.client.component.http.PostHttpRequestHandler;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.sdk.core.util.LogUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;

/**
 * 民生银行客户端
 *
 * @author
 * @version CmbcApiClient.java, v 0.1 2025-02-10 15:37:48
 */
@Slf4j
public class CmbcApiClient extends AbstractApiClient<CmbcBizRequest, CmbcBaseResponse, CmbcTradeApiDefinitionEnum> {
    /**
     * 请求执行器
     */
    protected IHttpRequestHandler httpRequestHandler = new PostHttpRequestHandler();

    private static Session session;

    private static SM2PrivateKey priKey;

    private static X509Cert payCompanyPublicKeyCert;

    private static X509Cert fubeiPrivateKeyCert;

    static {
        try {
            JCrypto.getInstance().initialize(JCrypto.JSOFT_LIB, null);
            session = JCrypto.getInstance().openSession(JCrypto.JSOFT_LIB);
        } catch (PKIException e) {
            e.printStackTrace();
        }
    }

    public CmbcApiClient(DefaultClientConfigModel apiClientConfig) throws FsApiException {
        super(apiClientConfig);

        CmbcClientConfigModel configModel = (CmbcClientConfigModel) apiClientConfig;
        try {
            String fubeiPrivateKeyPath = configModel.getFubeiPrivateKeyPath();
            String payCompanyPublicKeyPath = configModel.getPayCompanyPublicKeyPath();
            String password = configModel.getPassword();
            // 获取签名私钥
            priKey = getPrivateKeyFromSM2(fubeiPrivateKeyPath, password);
            // 获取民生银行公钥
            payCompanyPublicKeyCert = X509CertHelper.parse(payCompanyPublicKeyPath);
            // 获取付呗私钥
            fubeiPrivateKeyCert = CertUtil.getCertFromSM2(fubeiPrivateKeyPath);
        } catch (Exception e) {
            LogUtil.error(log, "CmbcApiClient >> 民生银行通道公私钥加载失败! ", e);
            throw new FsApiException("加载公私钥失败", e);
        }
    }

    @Override
    public CmbcBaseResponse execute(CmbcBizRequest request, CmbcTradeApiDefinitionEnum apiDefinition) throws FsApiException {
        return doExecute(request, apiDefinition);
    }

    @Override
    public CmbcBaseResponse execute(CmbcBizRequest request, CmbcTradeApiDefinitionEnum apiDefinition, DefaultClientConfigModel configModel) throws FsApiException {
        return doExecute(request, apiDefinition, configModel);
    }

    /**
     * 执行请求
     *
     * @param request
     * @param iApiDefinition
     * @param customConfig
     * @return
     * @throws FsApiException
     */
    @Override
    protected CmbcBaseResponse doExecute(CmbcBizRequest request, CmbcTradeApiDefinitionEnum iApiDefinition, DefaultClientConfigModel customConfig) throws FsApiException {
        long beginTime = System.currentTimeMillis();
        LogUtil.info(log, "{} >> 执行请求开始 >> iApiDefinition={}, request={}", getClientInfo().getClientDesc(), iApiDefinition, request);
        // 构建请求上下文
        DefaultRequestContext requestContext = buildRequestContext(iApiDefinition, request, customConfig);
        ApiRequestModel apiRequestModel = null;
        ApiResponseModel apiResponseModel = null;
        try {
            // 入参数校验
            checkParam(request, requestContext);
            // 构建请求数据
            apiRequestModel = buildApiRequestModel(request, requestContext);
            // 请求开始时间
            long reqBeginTime = System.currentTimeMillis();
            // 执行post请求
            apiResponseModel = httpRequest(apiRequestModel, requestContext);
            // 请求结束时间
            long reqEndTime = System.currentTimeMillis();
            Object requestBody = ObjectUtils.defaultIfNull(apiRequestModel.getRequestBody(), apiRequestModel.getRequestForm());
            LogUtil.debug(log, "{} >> 请求结束[密文] >> url={}, method={}, request={}, response={}, cost={}ms, reqcost={}ms",
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel.getApiURL(), iApiDefinition, requestBody,
                    apiResponseModel.getResponseBody(), System.currentTimeMillis() - beginTime, reqEndTime - reqBeginTime);

            CmbcBaseResponse response = buildApiResponse(apiResponseModel, apiRequestModel, requestContext);

            LogUtil.info(log, "{} >> 请求结束[明文] >> url={}, method={}, request={}, response={}, totalcost={}ms, reqcost={}ms",
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel.getApiURL(), requestContext.getIApiDefinition(), JSONObject.toJSONString(apiRequestModel.getRequest()),
                    JSONObject.toJSONString(apiResponseModel.getResponse()), System.currentTimeMillis() - beginTime, reqEndTime - reqBeginTime);
            // 反序列化构建响应结果
            return response;
        } catch (FsApiException e) {
            LogUtil.error(log, "{} >> 请求业务异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    iApiDefinition, request, apiRequestModel, apiResponseModel);
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 请求未知异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    iApiDefinition, request, apiRequestModel, apiResponseModel);
            LogUtil.info(log, "{} >> 请求未知异常[明文] >> url={}, method={}, request={}, cost={}ms", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel == null ? "" : apiRequestModel.getApiURL(),
                    requestContext.getIApiDefinition(),
                    JSONObject.toJSONString(apiRequestModel == null ? "" : apiRequestModel.getRequest()),
                    System.currentTimeMillis() - beginTime);
            throw new FsApiException(e.getMessage(), e);
        }
    }

    /**
     * 构建请求上下文
     *
     * @param tradeApiDefinitionEnum
     * @param request
     * @return
     */
    @Override
    protected DefaultRequestContext buildRequestContext(CmbcTradeApiDefinitionEnum tradeApiDefinitionEnum, CmbcBizRequest request, DefaultClientConfigModel customConfig) {

        DefaultRequestContext context = new DefaultRequestContext();
        context.setIApiDefinition(tradeApiDefinitionEnum);

        CmbcClientConfigModel clientConfig = (CmbcClientConfigModel) this.apiClientConfig;
        // 复制默认参数
        CmbcClientConfigModel config = BeanUtil.copyProperties(clientConfig, CmbcClientConfigModel.class);
        if (customConfig != null) {
            config.setAppId(StringUtils.isEmpty(customConfig.getAppId()) ? clientConfig.getAppId() : customConfig.getAppId());
        }
        context.setApiClientConfig(config);
        context.setClientInfoModel(getClientInfo());
        return context;
    }

    /**
     * 构建请求参数
     *
     * @param request
     * @param context
     * @return
     */
    @Override
    protected ApiRequestModel buildApiRequestModel(CmbcBizRequest request, DefaultRequestContext context) {
        // sdk客户端配置
        CmbcClientConfigModel configModel = (CmbcClientConfigModel) context.getApiClientConfig();

        ApiRequestModel apiRequestModel = new ApiRequestModel();
        // 设置网关地址
        apiRequestModel.setApiURL(context.getApiClientConfig().getApiParentURL() + context.getIApiDefinition().getApiURI());

        Map<String, Object> requestParams =  BeanUtil.beanToMap(request);
        requestParams.put("platformId", configModel.getAppId());
        String requestJsonStr = JSONObject.toJSONString(requestParams);
        String sign = getSign(requestJsonStr, configModel);
        JSONObject signContext = new JSONObject();
        signContext.put("sign", sign);
        signContext.put("body", requestJsonStr);

        String encryptContext = encrypt(signContext.toJSONString());
        JSONObject requestJson = new JSONObject();
        requestJson.put("businessContext", encryptContext);
        apiRequestModel.setRequestBody(requestJson.toJSONString());
        apiRequestModel.setRequest(request);
        return apiRequestModel;
    }

    /**
     * 处理客户端信息
     *
     * @return
     */
    @Override
    protected ClientInfoModel getClientInfo() {
        ClientInfoModel clientInfoModel = new ClientInfoModel();
        clientInfoModel.setClientName("民生银行");
        clientInfoModel.setClientCode("cmbc-sdk");
        return clientInfoModel;
    }

    @Override
    protected CmbcBaseResponse buildApiResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
        // 响应结果
        JSONObject resJson = JSONObject.parseObject(apiResponseModel.getResponseBody());
        // 响应结果
        CmbcBaseResponse response = resJson.toJavaObject(CmbcBaseResponse.class);
        // 业务密文
        String encryptedBody;
        if (response == null || StringUtils.isBlank(encryptedBody = response.getBusinessContext())) {
            return response;
        }
        String decryptContext = decrypt(encryptedBody);

        JSONObject resultJson = JSONObject.parseObject(decryptContext);
        String sign = resultJson.getString("sign");
        String body = resultJson.getString("body");
        // 如果需要验签
        if (!signCheck(sign, body)) {
            throw new FsApiException("验签失败");
        }
        // 业务响应结果
        CmbcBizResponse cmbcBizResponse = JSONObject.parseObject(body, requestContext.getIApiDefinition().getResponseClass());
        apiResponseModel.setResponse(cmbcBizResponse);
        response.setDecryptBusinessContext(cmbcBizResponse);
        return response;
    }

    @Override
    protected ApiResponseModel httpRequest(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws IOException {
        Map<String, String> headMap = new HashMap<>();
        headMap.put("Accept-Encoding", "identity");
        apiRequestModel.setHeadMap(headMap);
        return httpRequestHandler.httpRequest(apiRequestModel, requestContext);
    }

    /**
     * 加签
     *
     * @param context
     * @param configModel
     * @return
     */
    private static String getSign(String context, CmbcClientConfigModel configModel) {
        String sign = "";
        try {
            sign = new String(
                    new SignatureUtil2().p1SignMessage(Mechanism.SM3_SM2, context.getBytes(StandardCharsets.UTF_8), priKey, session));
        } catch (PKIException e) {
            e.printStackTrace();
        }
        return sign;
    }

    /**
     * 加密
     *
     * @param signContext 需要加密的报文
     * @return
     */
    @SuppressWarnings("deprecation")
    public static String encrypt(String signContext) {
        byte[] encryptedData = null;
        try {
            X509Cert[] certs = {payCompanyPublicKeyCert};
            encryptedData = EnvelopeUtil.envelopeMessage(signContext.getBytes(StandardCharsets.UTF_8), Mechanism.SM4_CBC, certs);
        } catch (PKIException e) {
            e.printStackTrace();
        }
        return new String(encryptedData, StandardCharsets.UTF_8);
    }

    /**
     * 解密
     *
     * @param encryptContext 需要解密的报文
     * @return
     */
    public static String decrypt(String encryptContext) {
        String decodeText = null;
        try {
            byte[] sourceData = EnvelopeUtil.openEvelopedMessage(encryptContext.getBytes(StandardCharsets.UTF_8), priKey, fubeiPrivateKeyCert, session);
            decodeText = new String(sourceData, StandardCharsets.UTF_8);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return decodeText;
    }

    /**
     * 验证签名
     *
     * @param sign 签名
     * @param body 参数明文
     * @return
     */
    @SuppressWarnings("unchecked")
    public static Boolean signCheck(String sign, String body) {
       boolean signCheckSuccess = false;
        try {
            signCheckSuccess = new SignatureUtil2().p1VerifyMessage(Mechanism.SM3_SM2, body.getBytes(StandardCharsets.UTF_8),
                    sign.getBytes(), payCompanyPublicKeyCert.getPublicKey(), session);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return signCheckSuccess;
    }

    public static SM2PrivateKey getPrivateKeyFromSM2(String sm2Path, String sm2PWD) throws PKIException {
        if (sm2Path == null) {
            throw new IllegalArgumentException("null not allowed for sm2Path");
        } else if (sm2PWD == null) {
            throw new IllegalArgumentException("null not allowed for sm2PWD");
        } else {
            FileInputStream fis = null;
            ASN1InputStream ais = null;

            SM2PrivateKey var8;
            try {
                fis = new FileInputStream(sm2Path);
                byte[] data = new byte[fis.available()];
                fis.read(data);
                fis.close();
                boolean isB64 = ASN1Parser.isBase64Encode(data);
                if (isB64) {
                    data = ASN1Parser.convertBase64(data);
                    data = Base64.decode(data);
                }

                ByteArrayInputStream bis = new ByteArrayInputStream(data);
                ais = new ASN1InputStream(bis);
                SM2Pfx object = SM2Pfx.getInstance(ais.readObject());
                var8 = object.getPrivateKey(sm2PWD);
            } catch (Exception var20) {
                throw new PKIException(var20.getMessage());
            } finally {
                if (ais != null) {
                    try {
                        ais.close();
                    } catch (Exception var19) {
                    }
                }

                if (fis != null) {
                    try {
                        fis.close();
                    } catch (Exception var18) {
                    }
                }

            }

            return var8;
        }
    }
}