/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package
        com.fshows.ccbpay.client.impl;

import cn.hutool.core.util.XmlUtil;
import cn.hutool.json.JSONObject;
import cn.hutool.json.XML;
import com.fshows.ccbpay.client.base.ApiClientConfig;
import com.fshows.ccbpay.client.base.ICcbPayApiDefinition;
import com.fshows.ccbpay.client.base.IHttpRequest;
import com.fshows.ccbpay.client.base.ISigner;
import com.fshows.ccbpay.enums.wlpt.CcbWlptApiDefinitionEnum;
import com.fshows.ccbpay.exception.CcbPayApiException;
import com.fshows.ccbpay.request.base.CcbPayBaseRequest;
import com.fshows.ccbpay.request.base.CcbWlptCommonRequest;
import com.fshows.ccbpay.response.base.CcbPayBaseResponse;
import com.fshows.ccbpay.util.CcbRequestUtils;
import com.fshows.ccbpay.util.LogUtil;
import com.fshows.ccbpay.util.ValidateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.Map;


/**
 * 建行开放平台交易接口客户端
 *
 * @author liluqing
 * @version UnionPayApiClientImpl.java, v 0.1 2022-03-04 15:02
 */
@Slf4j
public class CcbWlptApiClientImpl extends AbstractCcbPayApiClientImpl {

    public CcbWlptApiClientImpl(ApiClientConfig apiClientConfig) throws CcbPayApiException {
        this(apiClientConfig, new WlptRequestImpl(), new DefaultSignerImpl());
    }

    public CcbWlptApiClientImpl(ApiClientConfig apiClientConfig, IHttpRequest httpRequest) throws CcbPayApiException {
        super(apiClientConfig, httpRequest, null);
    }

    public CcbWlptApiClientImpl(ApiClientConfig apiClientConfig, IHttpRequest httpRequest, ISigner signer) throws CcbPayApiException {
        super(apiClientConfig, httpRequest, signer);
    }

    @Override
    public <T extends CcbPayBaseResponse, R extends ICcbPayApiDefinition> T execute(CcbPayBaseRequest<T, R> request, R tradeApiDefinitionEnum) throws CcbPayApiException {
        return super.doExecute(request, tradeApiDefinitionEnum);
    }

    /**
     * 添加对入参进行额外处理
     *
     * @param bizRequest             业务请求参数
     * @param tradeApiDefinitionEnum
     * @return
     */
    @Override
    protected CcbPayBaseRequest buildBizRequest(CcbPayBaseRequest bizRequest, ICcbPayApiDefinition tradeApiDefinitionEnum) {
        CcbWlptCommonRequest request = (CcbWlptCommonRequest) bizRequest;
        // 设置交易码
        request.setTxCode(((CcbWlptApiDefinitionEnum) tradeApiDefinitionEnum).getTxCode());
        request.setCustId(apiClientConfig.getMainMerchantNumber());
        request.setUserId(apiClientConfig.getUserId());
        request.setPassword(apiClientConfig.getUserPwd());
        request.setLanguage(apiClientConfig.getLanguage());
        return request;
    }

    /**
     * 接口参数序列化
     *
     * @param request
     * @param tradeApiDefinitionEnum
     * @return
     */
    @Override
    protected String serializableRequest(CcbPayBaseRequest request, ICcbPayApiDefinition tradeApiDefinitionEnum) {
        Map<String, String> requstMap = CcbRequestUtils.toMap(request);

        // 明细需要单独转
        StringBuilder xmlBuilder = new StringBuilder();
        Map<String, String> detailMap = CcbRequestUtils.toMap(((CcbWlptCommonRequest<?>) request).getTxInfo());
        for (Map.Entry<String, String> entry : detailMap.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();
            if (value != null) {
                xmlBuilder.append("<").append(key).append(">").append(value).append("</").append(key).append(">");
            }
        }
        requstMap.put("TX_INFO", xmlBuilder.toString());
        return CcbRequestUtils.generateXml(requstMap);
    }

    /**
     * 响应结果反序列化
     *
     * @param resBody
     * @param tradeApiDefinitionEnum
     * @return
     */
    @Override
    protected CcbPayBaseResponse parseResponse(String resBody, ICcbPayApiDefinition tradeApiDefinitionEnum) {
        try {
            JSONObject txRes = XML.toJSONObject(resBody);

            return com.alibaba.fastjson.JSONObject.parseObject(txRes.get("TX").toString(), tradeApiDefinitionEnum.getResponseClass());
        } catch (Exception e) {
            LogUtil.error(log, "【ccb-sdk】响应结果反序列化异常 >> resBody={}", e, resBody);
//            throw new CcbPayApiException("[ccb-sdk]响应结果反序列化异常");
        }

        return null;
    }

    @Override
    protected void checkParam(CcbPayBaseRequest bizRequest, ICcbPayApiDefinition apiDefinition) throws CcbPayApiException {
        ValidateUtil.notNull(bizRequest, "request请求参数不能为空");
        System.out.println();
        // 设置交易码
        CcbWlptCommonRequest<?> request = (CcbWlptCommonRequest<?>) bizRequest;
        boolean checkRequestClass = StringUtils.equals(request.getTxInfo().getClass().getCanonicalName(), apiDefinition.getRequestClass().getCanonicalName());
        if (!checkRequestClass) {
            throw new CcbPayApiException("请求参数类型不正确");
        }
        // 当校验开关打开时才开启主动前置参数校验
        if (this.apiClientConfig.isCheckParam()) {
            ValidateUtil.validateWithThrow(bizRequest);
        }
    }

    @Override
    protected String getServerURL(ICcbPayApiDefinition tradeApiDefinitionEnum) {
        return apiClientConfig.getApiParentURL();
    }
}