/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package
        com.fshows.ccbpay.client.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.codec.Base64;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.crypto.SecureUtil;
import cn.hutool.crypto.asymmetric.Sign;
import cn.hutool.crypto.digest.DigestAlgorithm;
import cn.hutool.crypto.digest.Digester;
import com.alibaba.fastjson.JSONObject;
import com.fshows.ccbpay.client.base.ApiClientConfig;
import com.fshows.ccbpay.client.base.ISigner;
import com.fshows.ccbpay.exception.CcbPayApiException;
import com.fshows.ccbpay.request.base.CcbPayBaseRequest;
import com.fshows.ccbpay.response.base.CcbPayBaseResponse;
import com.fshows.ccbpay.util.LogUtil;
import com.fshows.ccbpay.util.CcbPaySignUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * 建行开放平台加签默认实现
 *
 * @author liluqing
 * @version DefaultSignerImpl.java, v 0.1 2022-03-02 18:32
 */
@Slf4j
public class DefaultSignerImpl<T extends CcbPayBaseRequest, R extends CcbPayBaseResponse> implements ISigner<T, R> {

    @Override
    public String sign(T request, ApiClientConfig apiClientConfig) throws CcbPayApiException {
        // 执行加签动作
        return doSign(request, apiClientConfig);
    }

    @Override
    public Boolean verifySign(R response, ApiClientConfig apiClientConfig) throws CcbPayApiException {
        try {
            // 获取待加签字符串
            String waitSignStr = getWaitSignStr(response);
            // 获取签名字符串
            String signStr = response.getSignature();
            // 获取待加签字符串的数字摘要
            Digester sha256 = new Digester(DigestAlgorithm.SHA256);
            String digestHex = sha256.digestHex(waitSignStr);
            // 执行验签
            Sign sign = new Sign(
                    apiClientConfig.getSignTypeEnum().getAlgorithm(),
                    SecureUtil.decode(apiClientConfig.getFubeiPrivateKey()),
                    SecureUtil.decode(apiClientConfig.getCcbPayPublicKey()));
            return sign.verify(digestHex.getBytes(apiClientConfig.getCharset()), Base64.decode(signStr));
        } catch (Exception e) {
            LogUtil.error(log, "【ccbpay-sdk】建行开放平台响应结果验签失败 >> response={}", e, response);
            throw new CcbPayApiException("建行开放平台响应结果验签失败", e);
        }
    }

    /**
     * 执行方法加签
     *
     * @param obj
     * @param apiClientConfig
     * @return
     * @throws CcbPayApiException
     */
    public String doSign(T obj, ApiClientConfig apiClientConfig) throws CcbPayApiException {
        try {
            // 获取待加签字符串
            String waitSignStr = getWaitSignStr(obj);
//            Digester md5 = new Digester(DigestAlgorithm.SHA256);
//            // 获取待加签字符串的数字摘要
//            String digestHex = md5.digestHex(waitSignStr);
//            // 创建加签对象
//            Sign sign = new Sign(
//                    apiClientConfig.getSignTypeEnum().getAlgorithm(),
//                    SecureUtil.decode(apiClientConfig.getFubeiPrivateKey()),
//                    null);
//            // 使用付呗私钥进行加签
//            byte[] signed = sign.sign(digestHex.getBytes(apiClientConfig.getCharset()));
//            return Base64.encode(signed);
            Map<String, String> params = objectToMap(obj);
            params = CcbPaySignUtils.sign(params, apiClientConfig.getFubeiPrivateKey());
            return params.get("signature");
        } catch (Exception e) {
            LogUtil.error(log, "【ccbpay-sdk】建行开放平台请求加签失败 >> obj={}", e, obj);
            throw new CcbPayApiException("请求加签失败", e);
        }
    }

    /**
     * 获取待加签字符串
     *
     * @param obj
     * @return
     */
    public String getWaitSignStr(Object obj) {
        // bean转map
        Map<String, Object> paramMap = BeanUtil.beanToMap(obj, false, true);
        paramMap.remove("signature");
        List<String> keys = new ArrayList(paramMap.keySet());
        Collections.sort(keys);
        StringBuilder content = new StringBuilder();
        int index = 0;
        for(int i = 0; i < keys.size(); ++i) {
            String key = keys.get(i);
            Object value = paramMap.get(key);
            if (StringUtils.isNotBlank(key) && value != null) {
                content.append((index == 0 ? "" : "&") + key + "=" + value);
                ++index;
            }
        }
        return content.toString();
    }

    /**
     * 对象转换成map
     *
     * @param param
     * @return
     */
    private Map<String, String> objectToMap(Object param) throws IllegalAccessException {
        Map<String, String> map = Maps.newHashMap();
        Class<?> clazz = param.getClass();
        List<Field> fieldsList = new ArrayList<>();
        while (clazz != null) {
            Field[] declaredFields = clazz.getDeclaredFields();
            fieldsList.addAll(Arrays.asList(declaredFields));
            clazz = clazz.getSuperclass();
        }
        for (Field field : fieldsList) {
            field.setAccessible(true);
            // 若属性为null则不需要
            Object value = field.get(param);
            if (null == value) {
                continue;
            }
            // 为空字符串的则不需要
            if (ObjectUtil.isNull(value)) {
                continue;
            }
            String name = field.getName();
            // 去除序列化字段
            if ("serialVersionUID".equals(name)) {
                continue;
            }
            // 为基本类型
            if (this.simpleClass(field.getType())) {
                map.put(name, String.valueOf(value));
            } else {
                map.put(name, JSONObject.toJSONString(value));
            }
        }
        return map;
    }

    /**
     * 判断这个类是否是基础类型
     *
     * @param parameterType
     * @return
     */
    private boolean simpleClass(Class<?> parameterType) {
        return parameterType == String.class ||
                parameterType == Integer.class || parameterType == int.class ||
                parameterType == Double.class || parameterType == double.class ||
                parameterType == Long.class || parameterType == long.class ||
                parameterType == Float.class || parameterType == float.class ||
                parameterType == Byte.class || parameterType == byte.class ||
                parameterType == Boolean.class || parameterType == boolean.class ||
                parameterType == BigDecimal.class
                ;
    }

}