/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package
        com.fshows.ccbpay.client.impl;

import com.fshows.ccbpay.client.base.ApiClientConfig;
import com.fshows.ccbpay.client.base.IHttpRequest;
import com.fshows.ccbpay.client.base.ICcbPayApiClient;
import com.fshows.ccbpay.client.base.ICcbPayApiDefinition;
import com.fshows.ccbpay.client.base.ISigner;
import com.fshows.ccbpay.client.base.CcbPayHttpResult;
import com.fshows.ccbpay.exception.CcbPayApiException;
import com.fshows.ccbpay.request.base.CcbPayBaseRequest;
import com.fshows.ccbpay.request.base.CcbPayBizRequest;
import com.fshows.ccbpay.response.base.CcbPayBaseResponse;
import com.fshows.ccbpay.util.LogUtil;
import com.fshows.ccbpay.util.ValidateUtil;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;

/**
 * 默认建行开放平台api客户端实现
 *
 * @author liluqing
 * @version AbstractCcbPayApiClientImpl.java, v 0.1 2022-03-02 11:59
 */
@Setter
@Slf4j
public abstract class AbstractCcbPayApiClientImpl implements ICcbPayApiClient {

    /**
     * 客户端配置
     */
    protected ApiClientConfig apiClientConfig;

    /**
     * http请求器
     */
    protected IHttpRequest httpRequest;

    /**
     * 加签
     */
    protected ISigner<CcbPayBaseRequest, CcbPayBaseResponse> signer;

    public AbstractCcbPayApiClientImpl(ApiClientConfig apiClientConfig) throws CcbPayApiException {
        ValidateUtil.validateWithThrow(apiClientConfig);
        this.apiClientConfig = apiClientConfig;
        this.httpRequest = new DefaultHttpRequestImpl();
        this.signer = new DefaultSignerImpl();
    }

    public AbstractCcbPayApiClientImpl(ApiClientConfig apiClientConfig, IHttpRequest httpRequest, ISigner signer) throws CcbPayApiException {
        ValidateUtil.validateWithThrow(apiClientConfig);
        this.apiClientConfig = apiClientConfig;
        this.httpRequest = httpRequest;
        this.signer = signer;
    }

    /**
     * 执行建行开放平台请求
     *
     * @param bizRequest
     * @param tradeApiDefinitionEnum
     * @return
     * @throws CcbPayApiException
     */
    protected <R extends CcbPayBaseResponse, E extends ICcbPayApiDefinition> R doExecute(CcbPayBizRequest<R, E> bizRequest, ICcbPayApiDefinition tradeApiDefinitionEnum) throws CcbPayApiException {
        try {
            // 参数校验
            checkParam(bizRequest, tradeApiDefinitionEnum);
            // 补全全局公共参数
            CcbPayBaseRequest ccbpayBaseRequest = buildBizRequest(bizRequest, tradeApiDefinitionEnum);
            // 执行加签动作
            ccbpayBaseRequest.setSignature(signer.sign(ccbpayBaseRequest, this.apiClientConfig));
            // 序列化请求参数
            String body = serializableRequest(ccbpayBaseRequest, tradeApiDefinitionEnum);
            // 执行加签操作获取签名
            String url = getServerURL(tradeApiDefinitionEnum);
            // 执行HTTP post请求
            CcbPayHttpResult httpResult = requestPost(url, null, body);
            // 反序列化响应参数
            CcbPayBaseResponse response = parseResponse(httpResult.getBody(), tradeApiDefinitionEnum);
            // 根据是否验签配置判断是否需要验签
            if (this.apiClientConfig.isVrifySignResponse()) {
                boolean bo = signer.verifySign(response, this.apiClientConfig);
                if (!bo) {
                    LogUtil.error(log, "【ccbpay-sdk】响应结果验签失败 >> httpResult={}", httpResult);
                    throw new CcbPayApiException("[ccbpay-sdk]响应结果验签失败");
                }
            }
            // 反序列化响应参数
            return (R) response;
        } catch (CcbPayApiException e) {
            LogUtil.error(log, "【ccbpay-sdk】建行开放平台请求异常 >> tradeApiDefinition={}, bizRequest={}", e, tradeApiDefinitionEnum, bizRequest);
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "【ccbpay-sdk】建行开放平台请求异常 >> tradeApiDefinition={}, bizRequest={}", e, tradeApiDefinitionEnum, bizRequest);
            throw new CcbPayApiException(e.getMessage(), e);
        }
    }

    /**
     * 执行post请求
     *
     * @param url
     * @param authorization
     * @param requestBody
     * @return
     */
    protected CcbPayHttpResult requestPost(String url, String authorization, String requestBody) throws IOException, CcbPayApiException {
        long beginTime = System.currentTimeMillis();
        try {
            LogUtil.error(log, "【ccbpay-sdk】请求开始 >> url={}, authorization={}, request={}, begin={}", url, authorization, requestBody, beginTime);
            CcbPayHttpResult httpResult = httpRequest.post(url, authorization, requestBody, apiClientConfig);
            if (httpResult == null || StringUtils.isBlank(httpResult.getBody())) {
                throw new CcbPayApiException("ccbpay响应结果为空");
            }
            LogUtil.error(log, "【ccbpay-sdk】请求结束 >> url={}, request={}, response={}, cost={}ms", url, requestBody,
                    httpResult.getBody(), System.currentTimeMillis() - beginTime);
            return httpResult;
        } catch (CcbPayApiException e) {
            LogUtil.error(log, "【ccbpay-sdk】网络请求异常 >> url={}, request={}, cost={}ms",
                    e, url, requestBody, (System.currentTimeMillis() - beginTime));
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "【ccbpay-sdk】网络请求异常 >> url={}, request={}, cost={}ms",
                    e, url, requestBody, (System.currentTimeMillis() - beginTime));
            throw new CcbPayApiException("请求ccbpay接口异常", e);
        }
    }

    /**
     * 请求参数校验
     *
     * @param bizRequest
     * @return
     */
    protected void checkParam(CcbPayBizRequest bizRequest, ICcbPayApiDefinition apiDefinition) throws CcbPayApiException {
        ValidateUtil.notNull(bizRequest, "request请求参数不能为空");
        System.out.println();
        boolean checkRequestClass = StringUtils.equals(bizRequest.getClass().getCanonicalName(), apiDefinition.getRequestClass().getCanonicalName());
        if (!checkRequestClass) {
            throw new CcbPayApiException("请求参数类型不正确");
        }
        // 当校验开关打开时才开启主动前置参数校验
        if (this.apiClientConfig.isCheckParam()) {
            ValidateUtil.validateWithThrow(bizRequest);
        }
    }

    /**
     * 获取请求地址
     *
     * @param tradeApiDefinitionEnum
     * @return
     */
    protected String getServerURL(ICcbPayApiDefinition tradeApiDefinitionEnum) {
        return apiClientConfig.getApiParentURL() + tradeApiDefinitionEnum.getApiSubURI();
    }


    /**
     * 构建完整的请求参数
     *
     * @param request 业务请求参数
     * @param tradeApiDefinitionEnum
     * @return
     */
    protected abstract CcbPayBaseRequest buildBizRequest(
            CcbPayBizRequest request,
            ICcbPayApiDefinition tradeApiDefinitionEnum);


    /**
     * 序列化请求参数
     *
     * @param tradeApiDefinitionEnum
     * @return
     */
    protected abstract String serializableRequest(
            CcbPayBaseRequest request,
            ICcbPayApiDefinition tradeApiDefinitionEnum);

    /**
     * 解析响应参数
     *
     * @param tradeApiDefinitionEnum
     * @return
     */
    protected abstract CcbPayBaseResponse parseResponse(
            String resBody,
            ICcbPayApiDefinition tradeApiDefinitionEnum);
}