/**
 * fshows.com
 * Copyright (C) 2013-2020 All Rights Reserved.
 */
package com.fshows.ccbpay.util;

import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.StrUtil;
import lombok.extern.slf4j.Slf4j;

import java.io.UnsupportedEncodingException;
import java.security.KeyFactory;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Security;
import java.security.Signature;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Base64;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;

/**
 * 请求建行开放平台签名、验签
 *
 * @author wangjianguan
 * @version ccbPaySignUtils.java, v 0.1 2020-06-10 2:55 下午 wangjianguan
 */
@Slf4j
public class CcbPaySignUtils {

    /**
     * 签名
     */
    private static final String PARAM_SIGNATURE = "signature";
    /**
     * 字符接
     */
    private static final String DEFAULT_ENCODE = "UTF-8";
    private static final String PROVIDER_BC = "BC";


    static {
        if (Security.getProvider(PROVIDER_BC) == null) {
            Security.addProvider(new org.bouncycastle.jce.provider.BouncyCastleProvider());
        } else {
            Security.removeProvider(PROVIDER_BC);
            Security.addProvider(new org.bouncycastle.jce.provider.BouncyCastleProvider());
        }
    }


    /**
     * 签名
     *
     * @param params
     * @param privateKeyStr
     * @throws Exception
     */
    public static Map<String, String> sign(Map<String, String> params, String privateKeyStr) throws Exception {
        PrivateKey privateKey = getPrivateKey(privateKeyStr);
        return sign(params, privateKey, DEFAULT_ENCODE);
    }


    /**
     * 使用私钥签名
     *
     * @param params     参数
     * @param privateKey
     */
    private static Map<String, String> sign(Map<String, String> params, PrivateKey privateKey, String encode) throws Exception {
        // 校验设置默认字符集
        encode = StrUtil.isBlank(encode) ? DEFAULT_ENCODE : encode;
        // 过滤参数中的空格
        params = filterBlank(params);
        // 把map转成按照字典顺序的kv字符串
        String stringData = coverMap2String(params, true);
        LogUtil.info(log, "ccb pay activity sign params = {}", stringData);
        byte[] byteSign = null;
        String stringSign = null;
        try {
            byte[] signDigest = sha256X16(stringData, encode);
            byteSign = org.apache.commons.codec.binary.Base64.encodeBase64(signBySoft256(privateKey, signDigest));
        } catch (Exception e) {
            throw e;
        }
        stringSign = new String(byteSign, encode);
        LogUtil.info(log, "ccb pay activity sign sign = {}", stringSign);
        params.put(PARAM_SIGNATURE, stringSign);
        return params;
    }


    /**
     * 过滤请求报文中的空字符串或者空字符串
     *
     * @param content
     * @return
     */
    private static Map<String, String> filterBlank(Map<String, String> content) {
        Map<String, String> submitFromData = new HashMap<>();
        for (String key : content.keySet()) {
            String value = content.get(key);
            if (null != value && value.length() > 0) {
                // 对value值进行去除前后空处理
                submitFromData.put(key, value.trim());
            }
        }
        return submitFromData;
    }


    /**
     * map转string
     *
     * @param params          入参map
     * @param removeSignature true:忽略signature字段，false:拼接signature字段
     * @return 返回k=v&k=v拼接字符串
     */
    private static String coverMap2String(Map<String, String> params, boolean removeSignature) {
        if (MapUtil.isEmpty(params)) {
            return StrUtil.EMPTY;
        }
        TreeMap<String, String> treeMap = new TreeMap<>();
        Iterator<Map.Entry<String, String>> it = params.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<String, String> en = it.next();
            if (PARAM_SIGNATURE.equals(en.getKey()) && removeSignature) {
                continue;
            }
            treeMap.put(en.getKey(), en.getValue());
        }
        it = treeMap.entrySet().iterator();
        StringBuilder sf = new StringBuilder();
        while (it.hasNext()) {
            Map.Entry<String, String> en = it.next();
            sf.append(en.getKey()).append("=").append(en.getValue()).append("&");
        }
        return sf.substring(0, sf.length() - 1);
    }


    /**
     * Base64加密的秘钥字符串转私钥
     *
     * @param privateKeyStr
     * @return
     * @throws InvalidKeySpecException
     * @throws NoSuchAlgorithmException
     */
    private static PrivateKey getPrivateKey(String privateKeyStr)
            throws InvalidKeySpecException, NoSuchAlgorithmException {
        final byte[] priKeyBytes = Base64.getDecoder().decode(privateKeyStr);
        final PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(priKeyBytes);
        final KeyFactory keyFactory;
        try {
            keyFactory = KeyFactory.getInstance("RSA");
            return keyFactory.generatePrivate(keySpec);
        } catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            throw e;
        }
    }

    /**
     * @param privateKey
     * @param data
     * @return
     * @throws Exception
     */
    private static byte[] signBySoft256(PrivateKey privateKey, byte[] data) throws Exception {
        byte[] result = null;
        Signature st = Signature.getInstance("SHA256withRSA", "BC");
        st.initSign(privateKey);
        st.update(data);
        result = st.sign();
        return result;
    }

    /**
     * sha256计算后进行16进制转换
     *
     * @param data     待计算的数据
     * @param encoding 编码
     * @return 计算结果
     */
    private static byte[] sha256X16(String data, String encoding) throws UnsupportedEncodingException, NoSuchAlgorithmException {
        byte[] bytes = sha256(data.getBytes(encoding));
        StringBuilder sha256StrBuff = new StringBuilder();
        for (int i = 0; i < bytes.length; i++) {
            String str = Integer.toHexString(0xFF & bytes[i]);
            if (str.length() == 1) {
                sha256StrBuff.append("0");
            }
            sha256StrBuff.append(str);
        }
        try {
            return sha256StrBuff.toString().getBytes(encoding);
        } catch (UnsupportedEncodingException e) {
            throw e;
        }
    }


    private static byte[] sha256(byte[] data) throws NoSuchAlgorithmException {
        MessageDigest md = null;
        try {
            md = MessageDigest.getInstance("SHA-256");
            md.reset();
            md.update(data);
            return md.digest();
        } catch (Exception e) {
            throw e;
        }
    }


}