/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.ccbpay.request.trade.order;

import com.fshows.ccbpay.enums.trade.CcbPayTradeApiDefinitionEnum;
import com.fshows.ccbpay.response.trade.order.CcbQueryOrderStatusResponse;
import lombok.Data;

import javax.validation.constraints.NotBlank;
import org.hibernate.validator.constraints.Length;
import com.alibaba.fastjson.annotation.JSONField;

import com.fshows.ccbpay.request.base.CcbPayBizRequest;

/**
 * 订单状态轮询接口（在建行侧是轮询+入账操作）
 *
 * @author
 * @version CcbQueryOrderStatusRequest.java, v 0.1 2022-07-05 10:10:15
 */
@Data
public class CcbQueryOrderStatusRequest extends CcbPayBizRequest<CcbQueryOrderStatusResponse, CcbPayTradeApiDefinitionEnum> {

    private static final long serialVersionUID = -2213838555434149777L;


    /**
     * 商户号
     * 请联系开立商户的分行提供
     *
     * 示例：105421097080009
     * (必填)
     */
    @NotBlank
    @Length(max = 15, message = "merchantid长度不能超过15")
    @JSONField(name = "MERCHANTID")
    private String merchantid;

    /**
     * 柜台号
     * 请联系开立商户的分行提供
     *
     * 示例：902807340
     * (必填)
     */
    @NotBlank
    @Length(max = 9, message = "posid长度不能超过9")
    @JSONField(name = "POSID")
    private String posid;

    /**
     * 分行号
     * 请联系开立商户的分行提供（一级分行号）
     *
     * 示例：360000000
     * (必填)
     */
    @NotBlank
    @Length(max = 9, message = "branchid长度不能超过9")
    @JSONField(name = "BRANCHID")
    private String branchid;

    /**
     * 集团商户信息
     * （非集团商户无需上送）集团主商户代理发起子商户交易时（验证主商户柜台公钥），需将集团主商户信息填入该字段，格式为：JTSH:主商户号+主商户柜台+主商户分行号例如：JTSH:105000000000000123456789441000000其中：105000000000000为主商户号，123456789为主商户柜台号，441000000为主商户一级分行号
     *
     * 示例：JTSH:105000000000000123456789441000000
     * (特殊必填)
     */
    @Length(max = 40, message = "groupmch长度不能超过40")
    @JSONField(name = "GROUPMCH")
    private String groupmch;

    /**
     * 加密串
     * 加密串通过[4.2.请求报文加密方法]中的说明生成，加密源串参数包含以下浅蓝色底纹的参数，如：TXCODE=PAY101&amp;MERFLAG=1&amp;ORDERID=20180906101102&amp;QRYTIME=1&amp;QRCODETYPE=2
     *
     * 示例：ylQZnF9G57gKymsjC99NTUyqUwynqgeCywvYhT%2FhW1%2CFRewTw4t0nEk8Jz%2FKYI%0AY6Pa7rQPRMJ6nmkqO7B
     * (必填)
     */
    @NotBlank
    @Length(max = 6144, message = "ccbparam长度不能超过6144")
    @JSONField(name = "ccbParam")
    private String ccbparam;

    /**
     * 交易码
     * 固定值PAY101
     *
     * 示例：PAY101
     * (必填)
     */
    @NotBlank
    @Length(max = 6, message = "txcode长度不能超过6")
    @JSONField(name = "TXCODE")
    private String txcode;

    /**
     * 商户类型
     * 1：线上商户2：线下商户商户类型请与分行确认网络特约商户和龙商户属于线上商户，卡部商户若有柜台号则归属线上商户，若没有柜台号，则按线下商户处理。
     *
     * 示例：1
     * (必填)
     */
    @NotBlank
    @Length(max = 1, message = "merflag长度不能超过1")
    @JSONField(name = "MERFLAG")
    private String merflag;

    /**
     * 终端编号1
     * 商户类型为2时上送请联系开立商户的分行提供
     *
     * 示例：1054210970800090001
     * (特殊必填)
     */
    @Length(max = 19, message = "termno1长度不能超过19")
    @JSONField(name = "TERMNO1")
    private String termno1;

    /**
     * 终端编号2
     * 商户类型为2时上送请联系开立商户的分行提供
     *
     * 示例：12345678
     * (特殊必填)
     */
    @Length(max = 8, message = "termno2长度不能超过8")
    @JSONField(name = "TERMNO2")
    private String termno2;

    /**
     * 订单号
     * 订单标识，确保在商户系统中唯一
     *
     * 示例：20180906101102
     * (必填)
     */
    @NotBlank
    @Length(max = 30, message = "orderid长度不能超过30")
    @JSONField(name = "ORDERID")
    private String orderid;

    /**
     * 查询次数
     * 上送查询的次数，从1开始（每次加1）
     *
     * 示例：1
     * (必填)
     */
    @NotBlank
    @Length(max = 2, message = "qrytime长度不能超过2")
    @JSONField(name = "QRYTIME")
    private String qrytime;

    /**
     * 二维码类型
     * 1：龙支付2：微信3：支付宝4：银联5：网联如未上送QRCODE则此参数为必输
     *
     * 示例：2
     * (特殊必填)
     */
    @Length(max = 2, message = "qrcodetype长度不能超过2")
    @JSONField(name = "QRCODETYPE")
    private String qrcodetype;

    /**
     * 码信息（一维码、二维码）
     * 客户出示的付款码（现支持龙支付、支付宝、微信、银联系的付款码）。如无法正确获取二维码类型，可上送消费时扫描传输的码信息。QRCODE与QRCODETYPE参数必选其中一个上送。
     *
     * 示例：134737690209713402
     * (特殊必填)
     */
    @Length(max = 128, message = "qrcode长度不能超过128")
    @JSONField(name = "QRCODE")
    private String qrcode;

    /**
     * 备注1
     * 商户自定义，按需上送。在建行商户服务平台查询流水时可见，中文需使用UTF-8编码。需与消费接口上送值保持一致
     *
     * 示例：remark1
     * (特殊必填)
     */
    @Length(max = 30, message = "remark1长度不能超过30")
    @JSONField(name = "REMARK1")
    private String remark1;

    /**
     * 备注2
     * 商户自定义，按需上送。在建行商户服务平台查询流水时可见，中文需使用UTF-8编码。需与消费接口上送值保持一致
     *
     * 示例：remark2
     * (特殊必填)
     */
    @Length(max = 30, message = "remark2长度不能超过30")
    @JSONField(name = "REMARK2")
    private String remark2;

    /**
     * 子商户公众账号ID
     * 微信支付专有字段。如果子商户号没有绑定appid，只返回openid；如果子商户号绑定一个appid或多个appid，返回openid和sub_openid；子商户是否绑定appid，请联系开立商户的分行
     *
     * 示例：wx3b8bb3fad101bb07
     * (特殊必填)
     */
    @Length(max = 32, message = "subAppid长度不能超过32")
    @JSONField(name = "SUB_APPID")
    private String subAppid;

    /**
     * 返回信息位图
     * 共20位，位图，0或空-不返回，1-返回。第1位：是否返回OPENID和SUB_OPENID第2位：是否返回优惠金额相关字段第4位：是否返回支付详细信息字段第5位：是否返回微信优惠金额和详情字段第6位：是否返回支付宝优惠金额和支付宝单品券优惠信息和支付宝所有优惠券信息字段
     *
     * 示例：11000000000000000000
     * (特殊必填)
     */
    @Length(max = 20, message = "returnField长度不能超过20")
    @JSONField(name = "RETURN_FIELD")
    private String returnField;

}