/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.ccbpay.request.trade.pay;

import com.fshows.ccbpay.enums.trade.CcbPayTradeApiDefinitionEnum;
import com.fshows.ccbpay.response.trade.pay.CcbScanCardNormalPayResponse;
import lombok.Data;

import javax.validation.constraints.NotBlank;

import org.hibernate.validator.constraints.Length;
import com.alibaba.fastjson.annotation.JSONField;

import com.fshows.ccbpay.request.base.CcbPayBizRequest;

/**
 * 建行付款码支付请求参数
 *
 * @author
 * @version CcbScanCardPayRequest.java, v 0.1 2022-07-04 16:57:17
 */
@Data
public class CcbScanCardPayRequest extends CcbPayBizRequest<CcbScanCardNormalPayResponse, CcbPayTradeApiDefinitionEnum> {

    private static final long serialVersionUID = -2213838555526243886L;


    /**
     * 交易码
     * 固定值PAY100
     *
     * 示例：PAY100
     * (必填)
     */
    @NotBlank
    @Length(max = 6, message = "txcode长度不能超过6")
    @JSONField(name = "TXCODE")
    private String txcode;

    /**
     * 商户类型
     * 1：线上商户2：线下商户商户类型请与分行确认网络特约商户和龙商户属于线上商户，卡部商户若有柜台号则归属线上商户，若没有柜台号，则按线下商户处理。
     *
     * 示例：1
     * (必填)
     */
    @NotBlank
    @Length(max = 1, message = "merflag长度不能超过1")
    @JSONField(name = "MERFLAG")
    private String merflag;

    /**
     * 终端编号1
     * 商户类型为2时上送请联系开立商户的分行提供
     *
     * 示例：1054210970800090001
     * (特殊必填)
     */
    @Length(max = 19, message = "termno1长度不能超过19")
    @JSONField(name = "TERMNO1")
    private String termno1;

    /**
     * 终端编号2
     * 商户类型为2时上送请联系开立商户的分行提供
     *
     * 示例：12345678
     * (特殊必填)
     */
    @Length(max = 8, message = "termno2长度不能超过8")
    @JSONField(name = "TERMNO2")
    private String termno2;

    /**
     * 订单号
     * 订单标识，确保在商户系统中唯一
     *
     * 示例：20180906101102
     * (必填)
     */
    @NotBlank
    @Length(max = 30, message = "orderid长度不能超过30")
    @JSONField(name = "ORDERID")
    private String orderid;

    /**
     * 码信息（一维码、二维码）
     * 客户出示的付款码（现支持龙支付、支付宝、微信、银联系的付款码）
     *
     * 示例：134737690209713402
     * (必填)
     */
    @NotBlank
    @Length(max = 128, message = "qrcode长度不能超过128")
    @JSONField(name = "QRCODE")
    private String qrcode;

    /**
     * 订单金额
     * 该笔订单的资金总额
     *
     * 示例：0.01
     * (必填)
     */
    @NotBlank
    @JSONField(name = "AMOUNT")
    private String amount;

    /**
     * 商品名称
     * 商品名称。当RETURN_FIELD字段第三位上送1时，该字段将展示在客户微信/支付宝账单的商品名称字段，按微信文档要求1个中文算3个字节，1个英文或数字算1个字节，请注意不要超长。
     *
     * 示例：MobilePhone
     * (特殊必填)
     */
    @Length(max = 128, message = "proinfo长度不能超过128")
    @JSONField(name = "PROINFO")
    private String proinfo;

    /**
     * 备注1
     * 商户自定义，按需上送。在建行商户服务平台查询流水时可见，中文需使用UTF-8编码。如需轮询，在轮询接口中需同步上送
     *
     * 示例：remark1
     * (特殊必填)
     */
    @Length(max = 30, message = "remark1长度不能超过30")
    @JSONField(name = "REMARK1")
    private String remark1;

    /**
     * 备注2
     * 商户自定义，按需上送。在建行商户服务平台查询流水时可见，中文需使用UTF-8编码。如需轮询，在轮询接口中需同步上送
     *
     * 示例：remark2
     * (特殊必填)
     */
    @Length(max = 30, message = "remark2长度不能超过30")
    @JSONField(name = "REMARK2")
    private String remark2;

    /**
     * 分账信息一
     * 分账字段，需要分账时，FZINFO1必送，有需要时再送FZINFO2格式：（1）分账组信息，最少1组，最多5组但总长度不超过200字节，支付时需上送完整的全部分账组信息。FZINFO1和FZINFO2累加金额等于付款金额AMOUNT。（2）格式：分账类型代码!编号类型^收款编号一^账号类型^费用名称一（限5个汉字内）^金额^退款标志#编号类型^收款编号二^账号类型^费用名称二（限5个汉字）^金额^退款标志#——每组中各要素分别用“^”分隔符分开，分账信息组间用“#”分隔符分开，最后以“#”结束。（3）分账类型代码：21-汇总间接二清（资金流：持卡人-收单机构-收单商户-收款编号）、22-汇总直接二清（资金流：持卡人-收单机构-收款编号）、31-明细间接二清（资金流：持卡人-收单机构-收单商户-收款编号）、32-明细直接二清（资金流：持卡人-收单机构-收款编号）（4）编号类型：01-商户编号、02-终端号、03-账号，必填。（5）收款编号：商户编号或者终端编号或账号（最长27位），必填。（6）账号类型：01-本行对公、02-本行对私借记、03-合约账户、07-内部帐。编号类型为03时必填，其余编号类型无需填值。（7）金额：整数部分最长10位，小数部分最长2位，如1.23，必填，无需补空格。（8）退款标志取值用法：0-未退款，1-已退款；支付时送0，退款时送1；必填。需使用escape编码，escape前FZINFO1的总长度不超过200。银联支付暂不支持分账。
     *
     * 示例：21!01^105110070110001^^^1.23^0#01^105110070110002^^^2.46^0#
     * (特殊必填)
     */
    @Length(max = 200, message = "fzinfo1长度不能超过200")
    @JSONField(name = "FZINFO1")
    private String fzinfo1;

    /**
     * 分账信息二
     * 分账字段格式：（1）分账组信息，最少1组，最多5组但总长度不超过200字节，支付时需上送完整的全部分账组信息。FZINFO1和FZINFO2累加金额等于付款金额AMOUNT。（2）格式：编号类型^收款编号一^账号类型^费用名称一（限5个汉字）^金额^退款标志#编号类型^收款编号二^账号类型^费用名称二（限5个汉字）^金额^退款标志#——每组中各要素分别用“^”分隔符分开，分账信息组间用“#”分隔符分开。（3）编号类型：01-商户编号、02-终端号、03-账号，必填。（4）收款编号：商户编号或者终端编号或账号（最长27位），必填。（5）账号类型：01-本行对公、02-本行对私借记、03-合约账户、07-内部帐。编号类型为03时必填，其余编号类型无需填值。（6）金额：整数部分最长10位，小数部分最长2位，如1.23，必填，无需补空格。（7）退款标志取值用法：0-未退款，1-已退款；支付时送0，退款时送1；必填。需使用escape编码，escape前FZINFO2的总长度不超过200。银联支付暂不支持分账。
     *
     * 示例：01^105110070110001^^^1.23^0#01^105110070110002^^^2.46^0#
     * (特殊必填)
     */
    @Length(max = 200, message = "fzinfo2长度不能超过200")
    @JSONField(name = "FZINFO2")
    private String fzinfo2;

    /**
     * 子商户公众账号ID
     * 微信支付专有字段。如果子商户号没有绑定appid，只返回openid；如果子商户号绑定一个appid或多个appid，返回openid和sub_openid；子商户是否绑定appid，请联系开立商户的分行
     *
     * 示例：wx3b8bb3fad101bb07
     * (特殊必填)
     */
    @Length(max = 32, message = "subAppid长度不能超过32")
    @JSONField(name = "SUB_APPID")
    private String subAppid;

    /**
     * 返回信息位图
     * 共20位，位图，0或空-不返回，1-返回。第1位：是否返回OPENID和SUB_OPENID第2位：是否返回优惠金额相关字段第3位：是否将“商品名称PROINFO”的值上送支付宝/微信，以便展示在客户的支付宝/微信账单中1：上送0或空：不上送第4位：是否返回支付详细信息字段第5位：是否返回微信优惠金额和微信优惠详情字段第6位：是否返回支付宝优惠金额和支付宝单品券优惠信息和支付宝所有优惠券信息字段
     *
     * 示例：11000000000000000000
     * (特殊必填)
     */
    @Length(max = 20, message = "returnField长度不能超过20")
    @JSONField(name = "RETURN_FIELD")
    private String returnField;

    /**
     * 实名支付
     * 实名支付功能，包含类型、证件号、姓名三个子域（如果本字段出现，那么本字段包含的三个子域均需出现。字段说明见下方[3.1.1.1实名支付字段说明]
     *
     * 示例：字段说明见下方[3.1.1.1实名支付字段说明]格式如下：{"USERNAME":"张三","IDNUMBER":"330000000000000000","IDTYPE":" IDENTITY_CARD "}
     * (特殊必填)
     */
    @JSONField(name = "USERPARAM")
    private String userparam;

    /**
     * 商品详情
     * 微信单品优惠功能字段，详见下文说明[3.1.1.2微信单品优惠活动detail字段列表说明]
     *
     * 示例：详见下文说明[3.1.1.2单品优惠活动detail字段列表说明]格式如下：{"cost_price":1,"receipt_id":"wx123","goods_detail":[{"goods_id":"id1","wxpay_goods_id":"1001","goods_name":"iPhone6s 16G","quantity":1,"price":1},{"goods_id":"id2","wxpay_goods_id":"1002","goods_name":"iPhone6s 32G","quantity":1,"price":1}]}备注：Int型参数值不带双引号，String型参数值带双引号。
     * (特殊必填)
     */
    @JSONField(name = "detail")
    private String detail;

    /**
     * 订单优惠标记
     * 微信订单优惠标记，代金券或立减优惠功能的参数。若创建优惠活动时添加了标志，则商户下单时需要传入相同的标记，用户同时符合其他规则才能享受优惠。
     *
     * 示例：WXG
     * (特殊必填)
     */
    @Length(max = 32, message = "goodsTag长度不能超过32")
    @JSONField(name = "goods_tag")
    private String goodsTag;

    /**
     * 订单包含的商品列表信息
     * 支付宝单品券优惠功能字段，详见下方说明[3.1.1.3支付宝单品券优惠活动goods_detail_ali字段列表说明]
     *
     * 示例：详见下文说明[3.1.1.3支付宝单品券优惠活动goods_detail_ali字段列表说明]格式如下：[{"goods_id":"apple-01","goods_name":"ipad","quantity":1,"price":2000,"goods_category":"34543238","categories_tree":"124868003|126232002|126252004","body":"特价手机","show_url":"http://www.alipay.com/xxx.jpg"}]备注：Number型参数值不带双引号，String型参数值带双引号。
     * (特殊必填)
     */
    @JSONField(name = "goods_detail_ali")
    private String goodsDetailAli;

    /**
     * 订单详情信息
     * 商户列表数|商户实际数|商户编号%商品列表数#商品实际数#商品名称^商品金额^商品数量
     *
     * 示例：1|1|105421097080009%1#1#MobilePhone^0.01^1
     * (特殊必填)
     */
    @Length(max = 100, message = "ordrDtlInf长度不能超过100")
    @JSONField(name = "Ordr_Dtl_Inf")
    private String ordrDtlInf;

    /**
     * 网络交易平台简称
     * 网络交易平台简称
     *
     * 示例：CCBLONGPAY
     * (特殊必填)
     */
    @Length(max = 30, message = "ntwTxnpltfShrtnm长度不能超过30")
    private String ntwTxnpltfShrtnm;

    /**
     * 商户机具终端编号
     *
     */
    @Length(max = 30, message = "busTerminalId长度不能超过100")
    private String busTerminalId;


    /**
     * 公交间连项目名称
     *
     * 示例：huaNan_CCB
     */
    @Length(max = 30, message = "busProjectName长度不能超过512")
    private String busProjectName;


    /**
     * 公交路线名称
     *
     * 示例：huaNan_CCB
     */
    @Length(max = 30, message = "busLine长度不能超过512")
    private String busLine;
}