/**
 * fshows.com
 * Copyright (C) 2013-2021 All Rights Reserved.
 */
package com.fshows.ark.spring.boot.starter.core.mq.rocketmq.producer;

import com.fshows.ark.spring.boot.starter.core.mq.base.FsMessage;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.FsMessageSendContext;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.FsProducerInvoke;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.FsProducerModel;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.FsSendErrorCallback;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.FsSendResult;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.FsSendSuccessCallback;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.IFshowsProducer;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.IProducerConfigUpdateHandler;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.IProducerProxyFactory;
import com.fshows.ark.spring.boot.starter.core.mq.base.producer.TransactionMessageManage;
import com.fshows.ark.spring.boot.starter.enums.ProducerReturnTypeEnum;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

/**
 * 生产者动态代理类
 *
 * @author liluqing
 * @version RocketProducerProxy.java, v 0.1 2021-08-18 17:20
 */
@Slf4j
public class RocketProducerProxyHandler implements InvocationHandler, IProducerConfigUpdateHandler {

    /**
     * 延迟关闭的线程池
     */
    private static final ScheduledExecutorService delayedShutdownExecutor = Executors.newScheduledThreadPool(2, r -> {
        Thread t = new Thread(r, "producer-delayed-shutdown");
        t.setDaemon(true);
        return t;
    });
    /**
     * 延迟关闭时间（毫秒），默认15秒，大于消息发送超时时间
     */
    private static final long DELAYED_SHUTDOWN_MILLIS = 15000;
    /**
     * 生产者bean（使用AtomicReference保证线程安全）
     */
    private final AtomicReference<IFshowsProducer> producerRef;
    /**
     * 生产者工厂
     */
    private IProducerProxyFactory producerProxyFactory;

    /**
     * 事务消息管理器
     */
    private TransactionMessageManage transactionMessageManage;

    /**
     * 生产者接口元信息
     */
    private FsProducerModel fsProducerModel;

    /**
     * 保存接口中发送方法的源数据
     */
    private Map<Method, RocketSendMessageMethodModel> sendMessageMethodMap;

    public RocketProducerProxyHandler(IFshowsProducer producer, FsProducerModel fsProducerModel,
            TransactionMessageManage transactionMessageManage, IProducerProxyFactory producerProxyFactory) {
        this.producerRef = new AtomicReference<>(producer);
        this.sendMessageMethodMap = fsProducerModel.getSendMessageMethodModelMap();
        this.fsProducerModel = fsProducerModel;
        this.transactionMessageManage = transactionMessageManage;
        this.producerProxyFactory = producerProxyFactory;
    }

    /**
     * 线程安全地更新生产者实例
     *
     */
    @Override
    public void restart() {
        log.info("ark-spring-boot-starter >> 开始更新生产者实例！");
        // 重新创建一个新的生产者实例
        IFshowsProducer newProducer = producerProxyFactory.getFshowsProducer(fsProducerModel, transactionMessageManage);
        // 启动新实例
        newProducer.start();
        // 原子性地替换生产者实例
        IFshowsProducer oldProducer = producerRef.getAndSet(newProducer);

        // 旧的实例进行延迟关闭
        if (oldProducer != null) {
            log.info("ark-spring-boot-starter >> 生产者实例已更新，旧实例将在{}秒后关闭", DELAYED_SHUTDOWN_MILLIS / 1000);

            // 延迟关闭旧实例，确保正在处理的消息能够完成
            delayedShutdownExecutor.schedule(() -> {
                try {
                    oldProducer.shutdown();
                    log.info("ark-spring-boot-starter >> 旧生产者实例已成功关闭");
                } catch (Exception e) {
                    log.error("ark-spring-boot-starter >> 关闭旧生产者实例时发生异常", e);
                }
            }, DELAYED_SHUTDOWN_MILLIS, TimeUnit.MILLISECONDS);
        }
    }

    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        // 如果是object的默认实现,则调用该方法的默认实现,比如toString、equals等
        if (Object.class.equals(method.getDeclaringClass())) {
            return method.invoke(this, args);
        }

        // 获取当前的生产者实例
        IFshowsProducer producer = producerRef.get();
        if (producer == null) {
            throw new IllegalStateException("生产者实例为null，无法发送消息");
        }

        // 本次消息发送的元信息
        RocketSendMessageMethodModel sendMessageMethod = sendMessageMethodMap.get(method);
        // 构建本次消息发送的上下文对象
        FsMessageSendContext messageSendcontext = buildMessageSendContext(sendMessageMethod, args);
        // 执行消息发送处理
        FsSendResult sendResult = sendMessage(messageSendcontext, producer);
        // 判断当前方法返回值类型,如果是无返回值,则返回null
        return ProducerReturnTypeEnum.VOID.equals(sendMessageMethod.getReturnTypeEnum()) ? null : sendResult;
    }

    /**
     * 获取当前生产者实例
     *
     * @return
     */
    @Override
    public IFshowsProducer getProducer() {
        return producerRef.get();
    }

    /**
     * 获取事务消息管理器
     *
     * @return
     */
    @Override
    public TransactionMessageManage getTransactionMessageManage() {
        return transactionMessageManage;
    }

    /**
     * 获取生产者配置元信息
     */
    @Override
    public FsProducerModel getFsProducerModel() {
        return fsProducerModel;
    }

    /**
     * 消息发送处理
     *
     * @param messageSendcontext
     * @param producer
     * @return
     */
    private FsSendResult sendMessage(FsMessageSendContext messageSendcontext, IFshowsProducer producer) {
        FsSendResult sendResult = producer.send(messageSendcontext);
        return sendResult;
    }

    /**
     * 构建消息发送上下文
     *
     * @return
     */
    private FsMessageSendContext buildMessageSendContext(RocketSendMessageMethodModel rocketSendMessageMethodModel,
            Object[] args) {
        FsMessageSendContext context = new FsMessageSendContext();
        context.setRocketSendMessageMethodModel(rocketSendMessageMethodModel);
        context.setProducerSendTypeEnum(rocketSendMessageMethodModel.getSendTypeEnum());
        context.setProducerTypeEnum(rocketSendMessageMethodModel.getProducerTypeEnum());
        // 消息发送成功回调
        FsProducerInvoke producerInvoke = rocketSendMessageMethodModel.getParamTypeEnum().getFsProducerInvoke();
        FsSendSuccessCallback successCallback = producerInvoke.buildFsSendSuccessCallback(rocketSendMessageMethodModel,
                args);
        context.setSuccessCallback(successCallback);
        // 消息发送失败回调
        FsSendErrorCallback errorCallback = producerInvoke.buildFsSendErrorCallback(rocketSendMessageMethodModel, args);
        context.setErrorCallback(errorCallback);

        // 本次发送的消息内容
        FsMessage fsMessage = producerInvoke.buildFsMessage(rocketSendMessageMethodModel, args);
        context.setFsMessage(fsMessage);
        return context;
    }
}