/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.calcite.avatica.remote;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Properties;

public class HARemoteProtobufService extends ProtobufService {
    private static final Logger LOG = LoggerFactory.getLogger(HARemoteProtobufService.class);

    private final AvaticaHttpClient client;
    private final ProtobufTranslation translation;
    private final FailoverRetryPolicy retryPolicy;

    public HARemoteProtobufService(Properties connectionInfo, AvaticaHttpClient client, ProtobufTranslation translation, FailoverRetryPolicy retryPolicy) {
        if (client instanceof AvaticaCommonsHttpClientImpl) {
            this.client = new LindormHttpClient(connectionInfo, (AvaticaCommonsHttpClientImpl) client);
        } else {
            this.client = client;
        }
        this.translation = translation;
        this.retryPolicy = retryPolicy;
    }

    @Override
    public Response _apply(Request request) {
        final Response resp;
        byte[] response;
        try {
            while (true) {
                try {
                    response = client.send(translation.serializeRequest(request));
                    break;
                } catch (Exception e) {
                    LOG.warn("Retrying remote request " + retryPolicy.getRetries() + " times...", e);
                    if (!retryPolicy.awaitUntilNextRetry(e.getCause())) {
                        throw e;
                    }
                }
            }
        } catch (Exception e) {
            LOG.warn("Failed to execute remote request: {}", request);
            throw new RuntimeException(e);
        }

        try {
            resp = translation.parseResponse(response);
        } catch (IOException e) {
            LOG.warn("Failed to deserialize reponse to {}. '{}'", request,
                    new String(response, StandardCharsets.UTF_8));
            // Not a protobuf that we could parse.
            throw new RuntimeException(e);
        }

        // The server had an error, throw an Exception for that.
        if (resp instanceof ErrorResponse) {
            throw ((ErrorResponse) resp).toException();
        }

        return resp;
    }
}
