package com.aliyun.drc.clusterclient.partition;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.aliyun.drc.clusterclient.RegionContext;
import com.aliyun.drc.clusterclient.impl.ClientCluster;
import com.aliyun.drc.clustermanager.ClusterManager;
import com.aliyun.drc.clustermanager.Register;
import com.aliyun.drc.clustermanager.RegisteredInfo;
import com.aliyun.drc.regionmanager.RegionRouter;
import com.aliyun.drc.regionmanager.RegionRouterInfo;
import com.aliyun.drc.util.CipherUtils;
import com.aliyun.drc.util.AbnormalThreadHook;

import com.aliyun.drc.util.ReadManifest;
import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.message.BasicNameValuePair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.Charset;
import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

public class PartitionPool extends Thread {

    private final static Logger logger = LoggerFactory.getLogger(PartitionPool.class);

    private final static int KEEP_ALIVE_PERIOD = 10 * 1000;

    private volatile boolean exited;

    private String guid;

    private String ip;

    private String seq;

    private Register register;

    private final ClientCluster cluster;

    private Map<String, Partition> pool;

    private Map<Thread, AbnormalThreadHook> monitoredThreads;

    private RegionRouter regionRouter;

    private RegionRouterInfo regionRouterInfo;

    public PartitionPool(final RegionContext context, final ClientCluster cluster) {
        this.setName("DTS-Keep-Alive-Thread");
        this.cluster = cluster;
        this.regionRouter = new RegionRouter(context);
        this.pool = new ConcurrentHashMap<String, Partition>();
        this.monitoredThreads = new ConcurrentHashMap<Thread, AbnormalThreadHook>();
    }

    public void init() throws Exception {
        regionRouterInfo = regionRouter.getRegionRouterInfo(guid);
        if (regionRouterInfo == null) {
            logger.error("region router info is null, guid:" + guid);
            return;
        }
        ClusterManager cm = new ClusterManager(regionRouterInfo.getClusterUrl());
        register = new Register(cm);
        cluster.setRegionRouterInfo(regionRouterInfo);
    }

    public void addPartition(Partition partition) {
        partition.setClientCluster(cluster);
        partition.setRegister(register);
        partition.setRegionRouterInfo(regionRouterInfo);
        partition.setGuid(guid);
        partition.setIp(ip);
        partition.setSeq(seq);
        pool.put(partition.getName(), partition);
    }

    public void removePartition(String partition) {
        if(pool.containsKey(partition)) {
            pool.remove(partition);
        }
    }

    public Partition getPartition(String partition) {
        if(pool.containsKey(partition)) {
            return pool.get(partition);
        } else {
            return null;
        }
    }

    public void addMonitoredThread(Thread thread, AbnormalThreadHook hook) {
        monitoredThreads.put(thread, hook);
    }

    @SuppressWarnings({"rawtypes", "unchecked"})
    private void keepAlive() throws Exception {
        List<NameValuePair> query = new ArrayList<NameValuePair>();
        query.add(new BasicNameValuePair("ts", String.valueOf(System.currentTimeMillis())));
        query.add(new BasicNameValuePair("group", regionRouterInfo.getConsumerGroup()));
        query.add(new BasicNameValuePair("guid", guid));
        query.add(new BasicNameValuePair("ip", ip));
        query.add(new BasicNameValuePair("seq", seq));
        query.add(new BasicNameValuePair("consumer", regionRouterInfo.getUsername()));
        query.add(new BasicNameValuePair("password", regionRouterInfo.getPassword()));
        query.add(new BasicNameValuePair("maxconns", "10240"));

        logger.info("register client, ip:" + ip + ", seq:" + seq);
        Map<String, String> props = new HashMap<String, String>();
        String token = CipherUtils.encrypt(URLEncodedUtils.format(query, Charset.defaultCharset()));
        props.put("token", token);
        RegisteredInfo registeredInfo = register.registerClientAsIdle(props);
        List<String> list = new ArrayList<String>();
        if (registeredInfo.getIsSuccess()) {
            JSONArray jsonArray = registeredInfo.getData();
            if (jsonArray != null && !jsonArray.isEmpty()) {
                for (Object o : jsonArray) {
                    JSONObject partitionObject = new JSONObject((Map) o);
                    String partition = partitionObject.getJSONObject("partition").getString("name");
                    if (pool.get(partition) == null) {
                        logger.info("start new partition: " + partitionObject.toJSONString());
                        cluster.doStart(partitionObject);
                    }
                    list.add(partition);
                }
            }
        }
        if (!pool.isEmpty()) {
            Iterator<Entry<String, Partition>> iterator = pool.entrySet().iterator();
            while (iterator.hasNext()) {
                Entry<String, Partition> entry = iterator.next();
                if (!list.contains(entry.getKey())) {
                    iterator.remove();
                    cluster.doStop(entry.getKey());
                } else {
                    entry.getValue().sendHeartbeat();
                    entry.getValue().forceActAsConsumed();
                }
            }
        } else {
            logger.info("client partition is empty, wait partition balance");
        }
    }

    private void monitorThreadAlive() {
        Iterator<Entry<Thread, AbnormalThreadHook>> iterator = monitoredThreads.entrySet().iterator();
        while (iterator.hasNext()) {
            Entry<Thread, AbnormalThreadHook> entry = iterator.next();
            Thread thread = entry.getKey();
            if (!thread.isAlive()) {
                entry.getValue().notifyThreadFailed(thread);
                iterator.remove();
            }
        }
    }

    @Override
    public void run() {
        String SDKVersion = ReadManifest.getValue("Build-Version");
        logger.info("Welcome to start keep alive thread, SDK version is " + (SDKVersion != null ? SDKVersion : "4.6.27.12.0"));
        while (exited == false) {
            try {
                keepAlive();
                monitorThreadAlive();
                Thread.sleep(KEEP_ALIVE_PERIOD);
            } catch (InterruptedException e) {
                logger.warn("keep alive thread interrupted...");
                exited = true;
                break;
            } catch (Exception e) {
                logger.error("keep alive thread exception: ", e);
            }
        }
    }

    public void shutdown() throws InterruptedException {
        this.exited = true;
        this.interrupt();
        this.join();
        logger.info("partition pool has been shutdown...");
    }

    public void setGuid(final String guid) {
        this.guid = guid;
    }

    public final String getGuid() {
        return guid;
    }

    public void setIp(final String ip) {
        this.ip = ip;
    }

    public final String getIp() {
        return ip;
    }

    public String getSeq() {
        return seq;
    }

    public void setSeq(String seq) {
        this.seq = seq;
    }
}