package com.aliyun.drc.clusterclient.partition;

import com.aliyun.drc.client.message.DataMessage.Record;
import com.aliyun.drc.clusterclient.impl.ClientCluster;
import com.aliyun.drc.clustermanager.CommittedInfo;
import com.aliyun.drc.clustermanager.Register;
import com.aliyun.drc.clustermanager.RegisteredInfo;
import com.aliyun.drc.regionmanager.RegionRouterInfo;
import com.aliyun.drc.util.CheckpointLinkedList;
import com.aliyun.drc.util.CipherUtils;
import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.message.BasicNameValuePair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class PartitionImpl implements Partition {

    private static Logger logger = LoggerFactory.getLogger(PartitionImpl.class);

    private String guid;

    private String ip;

    private String seq;

    private String partition;

    private String topic;

    private Register register;

    private ClientCluster cluster;

    private RegionRouterInfo regionRouterInfo;

    private CheckpointLinkedList<Record> ckList = new CheckpointLinkedList<Record>();

    public PartitionImpl(final String name) {
        partition = name;
    }

    /**
     * Setter for partition's attributes
     */
    public void setClientCluster(ClientCluster cluster) {
        this.cluster = cluster;
    }

    public void setRegister(Register register) {
        this.register = register;
    }

    public void setRegionRouterInfo(RegionRouterInfo regionRouterInfo) {
        this.regionRouterInfo = regionRouterInfo;
    }

    public void setGuid(final String guid) {
        this.guid = guid;
    }

    public void setTopic(String topic) {
        this.topic = topic;
    }

    public void setIp(final String ip) {
        this.ip = ip;
    }

    public void setSeq(String seq) {
        this.seq = seq;
    }

    public String getName() {
        return partition;
    }

    public void ackAsConsumed(Record record) {
        ckList.removeElementFromCheckpointQueue(record);
    }

    public void sendHeartbeat() {
        try {
            List<NameValuePair> query = new ArrayList<NameValuePair>();
            query.add(new BasicNameValuePair("ts", String.valueOf(System.currentTimeMillis())));
            query.add(new BasicNameValuePair("group", regionRouterInfo.getConsumerGroup()));
            query.add(new BasicNameValuePair("guid", guid));
            query.add(new BasicNameValuePair("ip", ip));
            query.add(new BasicNameValuePair("seq", seq));
            query.add(new BasicNameValuePair("consumer", regionRouterInfo.getUsername()));
            query.add(new BasicNameValuePair("password", regionRouterInfo.getPassword()));
            query.add(new BasicNameValuePair("partition", partition));
            query.add(new BasicNameValuePair("topic", topic));

            logger.info("sending partition heartbeat, group:" + regionRouterInfo.getConsumerGroup() + ", guid:" + guid + ", partition:" + partition);
            Map<String, String> props = new HashMap<String, String>();
            String token = CipherUtils.encrypt(URLEncodedUtils.format(query, Charset.defaultCharset()));
            props.put("token", token);
            RegisteredInfo response = register.registerClientAsActive(props);
            if (response.getIsSuccess() == false) {
                logger.warn("Get error #" + response.getErrCode() + " " + response.getErrMsg() + "when sending heartbeat: " + query);
            }
        } catch (Exception e) {
            logger.warn("sending heartbeat exception. ", e);
        }
    }

    public void forceActAsConsumed() throws Exception {
        Record record = ckList.getMin();
        if (record == null) {
            return;
        }
        try {
            Checkpoint checkpoint = new Checkpoint(record);
            List<NameValuePair> query = new ArrayList<NameValuePair>();
            query.add(new BasicNameValuePair("ts", String.valueOf(System.currentTimeMillis())));
            query.add(new BasicNameValuePair("group", regionRouterInfo.getConsumerGroup()));
            query.add(new BasicNameValuePair("guid", guid));
            query.add(new BasicNameValuePair("ip", ip));
            query.add(new BasicNameValuePair("seq", seq));
            query.add(new BasicNameValuePair("consumer", regionRouterInfo.getUsername()));
            query.add(new BasicNameValuePair("password", regionRouterInfo.getPassword()));
            query.add(new BasicNameValuePair("partition", partition));
            query.add(new BasicNameValuePair("offset", checkpoint.toString()));

            logger.info("sending partition commit, partition:" + partition + ", checkpoint:" + checkpoint.toString() + ", size:" + ckList.size());
            Map<String, String> props = new HashMap<String, String>();
            String token = CipherUtils.encrypt(URLEncodedUtils.format(query, Charset.defaultCharset()));
            props.put("token", token);
            CommittedInfo response = register.commit(props);
            if (response.getIsSuccess() == false) {
                logger.warn("Get error #" + response.getErrCode() + " " + response.getErrMsg() + "when sending commit: " + query);
                if (response.getErrCode() == 401 || response.getErrCode() == 443) {
                    cluster.doStop(partition);
                }
            }
        } catch (Exception e) {
            logger.warn("send partition commit exception. ", e);
        }
    }

    @Override
    public void pushRecordToPartition(Record record) {
        switch (record.getOpt()) {
            case HEARTBEAT:
                ckList.pushHeartBeatElement(record);
                break;
            default:
                ckList.putElementCheckpointQueue(record);
                break;
        }
    }

    public boolean equals(Object partitionImpl) {
        return ((PartitionImpl) partitionImpl).getName().equals(partition);
    }

    public int hashCode() {
        return partition.hashCode();
    }

}