package com.aliyun.drc.clusterclient.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

import com.aliyun.drc.clusterclient.ClusterListener;
import com.aliyun.drc.clusterclient.partition.Checkpoint;
import com.aliyun.drc.clusterclient.partition.Partition;
import org.slf4j.Logger;

import com.aliyun.drc.client.Listener;
import com.aliyun.drc.client.message.DataMessage;
import com.aliyun.drc.client.message.DataMessage.Record;
import com.aliyun.drc.clusterclient.message.ClusterMessage;

import static com.aliyun.drc.clusterclient.impl.DataTypeEnum.*;
import org.slf4j.LoggerFactory;

public class DrcClientListener implements Listener {

    private static Logger logger = LoggerFactory.getLogger(DrcClientListener.class);

    private static final long LOG_TAGGING_PERIOD_IN_MS = 60 * 1000;

    private Partition partition;

    private ClusterListener clusterListener;

    private ClientCluster clientCluster;

    private DataTypeEnum dataTypeEnum;

    private long lastStatisticsTime = 0L;

    public DrcClientListener(ClusterListener clusterListener, ClientCluster clientCluster, String dataType) {
        this.clusterListener = clusterListener;
        this.clientCluster = clientCluster;
        if (dataType.equalsIgnoreCase("dml")) {
            dataTypeEnum = DataTypeEnum.DML;
        } else {
            dataTypeEnum = DataTypeEnum.ALL;
        }
    }

    private void logTaggingInPeriod(Record record) {
        long currentTime = System.currentTimeMillis();
        if(lastStatisticsTime <= 0) {
            lastStatisticsTime = currentTime;
        }
        if(currentTime - lastStatisticsTime >= LOG_TAGGING_PERIOD_IN_MS) {
            lastStatisticsTime = currentTime;
            Checkpoint checkpoint = new Checkpoint(record);
            logger.info("record pushed in memory:" + checkpoint.toString() + ", msgQueue size:" + clusterListener.getMessageQueueSize());
        }
    }

    @Override
    public void notify(DataMessage message) throws Exception {
        List<ClusterMessage> messages = new ArrayList<ClusterMessage>();
        for (Record record : message.getRecordList()) {
            // record pushed in memory and msgQueue size log tagging in every 60s
            logTaggingInPeriod(record);

            switch (record.getOpt()) {
                // HEARTBEAT is only used for pushing checkpoint forward,
                // not used for user code callback
                case HEARTBEAT:
                    partition.pushRecordToPartition(record);
                    break;
                case DDL:
                    if (dataTypeEnum == DataTypeEnum.DML) {
                        break;
                    }
                default:
                    ClusterMessage msg = new ClusterMessage(record, partition);
                    messages.add(msg);
                    partition.pushRecordToPartition(record);
            }
        }
        if (messages.size() == 0) {
            return;
        }
        /*
        long needSleepTime = clusterListener.setNotifiedPartition(partition);
        if (needSleepTime > 0) {
            TimeUnit.MILLISECONDS.sleep(needSleepTime);
        }
        clusterListener.notifyWithoutHeartbeat(messages);
        */
        long timeNeedSleep = clusterListener.notifyMessages(partition, messages);
        if(timeNeedSleep > 0) {
            TimeUnit.MILLISECONDS.sleep(timeNeedSleep);
        }
    }

    @Override
    public void notifyRuntimeLog(String level, String log) throws Exception {
        if(level.equalsIgnoreCase("INFO")) {
            logger.info(log);
        } else if(level.equalsIgnoreCase("ERROR")) {
            logger.error(log);
        } else if(level.equalsIgnoreCase("WARN")) {
            logger.warn(log);
        } else {
            logger.warn(log);
        }
    }

    @Override
    public void handleException(Exception e) {
        try {
            logger.warn("Handle DrcClientListener exception, stop partition...", e);
            // clientCluster.doStop(partition.getName());
            Thread doStopThread = new Thread(new Runnable() {
                public void run() {
                    clientCluster.doStop(partition.getName());
                }
            });
            doStopThread.setName("DTS-DoStop-Thread");
            doStopThread.start();
            doStopThread.join();
        } catch (Exception e1) {
            logger.error("DrcClientListener do stop exception.", e1);
        }
    }

    public ClusterListener getListener() {
        return clusterListener;
    }

    public void setPartition(Partition partition) {
        this.partition = partition;
    }

    public Partition getPartition() {
        return partition;
    }
}