package com.aliyun.drc.clusterclient.impl;

import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;

import com.alibaba.fastjson.JSONObject;
import com.aliyun.drc.clusterclient.ClusterListener;
import com.aliyun.drc.regionmanager.RegionRouterInfo;
import com.aliyun.drc.util.ThreadMonitor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.aliyun.drc.client.DRCClient;
import com.aliyun.drc.client.DRCClientFactory;
import com.aliyun.drc.client.DataFilter;
import com.aliyun.drc.clusterclient.partition.Checkpoint;
import com.aliyun.drc.clusterclient.partition.Partition;
import com.aliyun.drc.clusterclient.partition.PartitionImpl;
import com.aliyun.drc.clusterclient.partition.PartitionPool;
import com.aliyun.drc.util.AbnormalThreadHook;

public class ClientCluster implements AbnormalThreadHook {

    private static Logger logger = LoggerFactory.getLogger(ClientCluster.class);

    private String guid;

    //private volatile boolean suspend;

    private int listenerIndex = 0;

    private List<ClusterListener> listeners;

    private Map<String, DrcClientListener> clientListeners;

    private Map<String, DRCClient> clientPools;

    private Map<Thread, String> clientThreadPools;

    private PartitionPool partitionPool;

    private RegionRouterInfo regionRouterInfo;

    public ClientCluster() {
        clientListeners = new ConcurrentHashMap<String, DrcClientListener>();
        clientPools = new ConcurrentHashMap<String, DRCClient>();
        clientThreadPools = new ConcurrentHashMap<Thread, String>();
    }

    public void doStart(JSONObject jsonObject) throws Exception {
        String partitionName = jsonObject.getJSONObject("partition").getString("name");
        String topic = jsonObject.getString("topic");
        String filter = (String) jsonObject.getJSONArray("tables").get(0);
        String offset = jsonObject.getString("offset");
        logger.info("Do start command, partition:" + partitionName + ", topic:" + topic + ", filter:" + filter + ", offset:" + offset);
        if(partitionPool.getPartition(partitionName) != null) {
            logger.warn("Partition " + partitionName + " already exists in PartitionPool, start command is ignored...");
            throw new Exception("Partition " + partitionName + " already exists in PartitionPool");
        }

        Partition partition = new PartitionImpl(partitionName);
        partition.setTopic(topic);
        ClusterListener clusterListener = getNextListener();
        DrcClientListener listener = new DrcClientListener(clusterListener, this, regionRouterInfo.getDataType());
        listener.setPartition(partition);

        Properties properties = new Properties();
        properties.put("manager.host", regionRouterInfo.getClusterUrl());
        properties.put("guid", guid);
        DRCClient client = DRCClientFactory.create(DRCClientFactory.Type.MYSQL, properties);
        StringBuffer dataFilterStr = new StringBuffer();
        for (String part : filter.split("\\|")) {
            dataFilterStr.append(part).append(".*|");
        }
        client.addDataFilter(new DataFilter(dataFilterStr.toString()));
        client.addListener(listener);

        Checkpoint checkpoint = new Checkpoint(offset);
        com.aliyun.drc.client.impl.Checkpoint drcClientCheckpoint = new com.aliyun.drc.client.impl.Checkpoint();
        if (checkpoint.getInstance() != null) {
            drcClientCheckpoint.setServerId(checkpoint.getInstance());
        }
        if (checkpoint.getFilePosition() != null) {
            drcClientCheckpoint.setPosition(checkpoint.getFilePosition());
        }
        if (checkpoint.getTimestamp() != null) {
            drcClientCheckpoint.setTimestamp(checkpoint.getTimestamp());
        }
        if (checkpoint.getId() != null) {
            drcClientCheckpoint.setRecordId(checkpoint.getId());
        }
        // use public ip for client
        if (regionRouterInfo.isUsePublicIp()) {
            client.usePublicIp();
        }
        client.initService(regionRouterInfo.getUsername(), topic, regionRouterInfo.getPassword(), drcClientCheckpoint, null);
        Thread serviceThread = client.startService();
        serviceThread.setName("DTS-DRCClient-" + partitionName + "-Thread");
        serviceThread.setUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {
            @Override
            public void uncaughtException(Thread arg0, Throwable arg1) {
                logger.error(arg0.getName() + ", " + arg1.toString());
            }
        });

//        if (suspend) {
//            client.suspend();
//        }
        // fill in data structure until client service is started successfully
        clusterListener.addListenedPartition(listener, partition);
        partitionPool.addPartition(partition);
        partitionPool.addMonitoredThread(serviceThread, this);
        clientListeners.put(partitionName, listener);
        clientPools.put(partitionName, client);
        clientThreadPools.put(serviceThread, partitionName);
        logger.info("Do start command finished, partition:" + partitionName + ", topic:" + topic + ", filter:" + filter + ", offset:" + offset);
    }

    public void doStop(String partition) {
        logger.info("Do stop command, partition:" + partition);
        partitionPool.removePartition(partition);
        DrcClientListener removedClientListener = clientListeners.get(partition);
        if(removedClientListener != null) {
            removedClientListener.getListener().removeListenedPartition(removedClientListener, removedClientListener.getPartition());
            clientListeners.remove(partition);
        }

        DRCClient client = clientPools.get(partition);
        if (client != null) {
            try {
                client.stopService();
                clientPools.remove(partition);
            } catch (Exception e) {
                logger.error("stop drcclient service exception: " + partition, e);
            }
        } else {
            logger.warn(partition + " find null drcclient");
        }
        Iterator<Entry<Thread, String>> iterator = clientThreadPools.entrySet().iterator();
        while (iterator.hasNext()) {
            Entry<Thread, String> entry = iterator.next();
            if (entry.getValue().equalsIgnoreCase(partition)) {
                if(!entry.getKey().isAlive()) {
                    iterator.remove();
                } else {
                    logger.error(entry.getKey().getName() + " still alive, stop drcclient service failed");
                }
                break;
            }
        }
        logger.info("Do stop command finished, partition:" + partition);
    }

    public void notifyThreadFailed(Thread t) {
        final String partition = clientThreadPools.get(t);
        if (partition != null) {
            logger.warn("monitor DRCClient thread is not alive, execute do stop... " + t.getName());
            doStop(partition);
        }
    }

    public void shutdown() throws Exception {
        Iterator<Entry<String, DRCClient>> iterator = clientPools.entrySet().iterator();
        while(iterator.hasNext()) {
            Entry<String, DRCClient> entry = iterator.next();
            try {
                entry.getValue().stopService();
                iterator.remove();
            } catch (Exception e) {
                logger.error("stop drcclient service exception: " + entry.getKey(), e);
            }
        }
        checkNotAliveThreads();
        logger.info("cluster has been shutdown...");
    }

    private void checkNotAliveThreads() throws Exception {
        int notAlive = 0;
        for(Thread t : clientThreadPools.keySet()) {
            logger.info("service Thread:" + t.getName() + ", current caller Thread:" + Thread.currentThread().getName());
            if(t != Thread.currentThread() && !t.isAlive()) {
                notAlive ++;
                logger.info(t.getName() + " is NOT alive, stop service success");
            } else {
                logger.warn(t.getName() + " is STILL alive, stop service failure");
            }
        }
        if(notAlive != clientThreadPools.size()) {
            logger.error("cluster shutdown failure, still has alive threads, total:" + clientThreadPools.size() + ", alive:" + (clientThreadPools.size() - notAlive));
            throw new Exception("cluster shutdown failure, still has alive threads");
        }
    }

    private ClusterListener getNextListener() {
        if (listenerIndex >= listeners.size()) {
            listenerIndex = 0;
        }
        return listeners.get(listenerIndex++);
    }

    public void setGuid(final String guid) {
        this.guid = guid;
    }

    public void setListeners(final List<ClusterListener> listeners) {
        this.listeners = listeners;
    }

    public void setPartitions(PartitionPool partitionPool) {
        this.partitionPool = partitionPool;
    }

    public void setRegionRouterInfo(RegionRouterInfo regionRouterInfo) {
        this.regionRouterInfo = regionRouterInfo;
    }

//    public void suspendClient() {
//        suspend = true;
//        for (Map.Entry<String, DRCClient> entry : clientPools.entrySet()) {
//            entry.getValue().suspend();
//            logger.info("partition:" + entry.getKey() + ",suspend");
//        }
//    }
//
//    public void resumeClient() {
//        suspend = false;
//        for (Map.Entry<String, DRCClient> entry : clientPools.entrySet()) {
//            entry.getValue().resume();
//            logger.info("partition:" + entry.getKey() + ",resume");
//        }
//    }
}