package com.aliyun.drc.clusterclient;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicInteger;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.aliyun.drc.clusterclient.impl.DrcClientListener;
import com.aliyun.drc.clusterclient.message.ClusterMessage;
import com.aliyun.drc.clusterclient.partition.Partition;

public abstract class ClusterListener extends Thread {

    private final static Logger logger = LoggerFactory.getLogger(ClusterListener.class);

    private static final long MAX_OCCUPY_TIME_IN_MS = 2000;

    private static final long DEFAULT_WAIT_TIME_IN_MS = 10;

    private volatile boolean exited = false;

    private long beginTime = 0L;

    private AtomicInteger count = new AtomicInteger(0);

    private Partition lastNotifiedPartition;

    private List<Partition> listenedPartitions = new ArrayList<Partition>();

    private List<DrcClientListener> listeners = new ArrayList<DrcClientListener>();

    private BlockingQueue<List<ClusterMessage>> msgQueue = new LinkedBlockingQueue<List<ClusterMessage>>(4096);

    /**
     * Abstract method notify for user code callback
     * @param messages
     * @throws Exception
     */
    public abstract void notify(List<ClusterMessage> messages) throws Exception;

    /**
     * Abstract method noException for user code callback
     * @param e
     */
    public abstract void noException(Exception e);

    @Override
    public void run() {
        this.setName("Notify-Message-Thread");
        while(!exited) {
            try {
                List<ClusterMessage> messages = msgQueue.take();
                notify(messages);   // user code callback
            } catch(InterruptedException e) {
                logger.warn("notify messages thread interrupted...");
                noException(e);
            } catch (Exception e) {
                logger.error("notify messages thread exception: ", e);
                noException(e);
            }
        }
    }

    synchronized public long notifyMessages(final Partition partition, List<ClusterMessage> messages) {
        try {
            msgQueue.put(messages);
        } catch(InterruptedException e) {
            logger.warn("put messages into msgQueue interrupted...");
        } catch(Exception e) {
            logger.error("put messages into msgQueue exception: ", e);
        }

        // only one partition, no need to sleep
        if(count.longValue() <= 1) {
            return 0;
        }
        if (beginTime <= 0 || lastNotifiedPartition == null || lastNotifiedPartition != partition) {
            lastNotifiedPartition = partition;
            beginTime = System.currentTimeMillis();
            return 0;
        } else {
            long currTime = System.currentTimeMillis();
            if (currTime - beginTime >= MAX_OCCUPY_TIME_IN_MS) {
                beginTime = 0;
                return DEFAULT_WAIT_TIME_IN_MS;
            } else {
                return 0;
            }
        }
    }

    @Deprecated
    public synchronized void notifyWithoutHeartbeat(List<ClusterMessage> messages) throws Exception {
        if (beginTime == 0) {
            beginTime = System.currentTimeMillis();
        }
        try {
            notify(messages);
        } catch (Exception e) {
            noException(e);
        }
    }

    @Deprecated
    public long setNotifiedPartition(final Partition partition) {
        if (count.longValue() <= 1) {
            // only one partition listener, no compete
            beginTime = -1;
            return 0;
        }

        if (beginTime <= 0 || lastNotifiedPartition == null || lastNotifiedPartition != partition) {
            lastNotifiedPartition = partition;
            beginTime = 0;
            return 0;
        } else {
            long currTime = System.currentTimeMillis();
            if (currTime - beginTime >= MAX_OCCUPY_TIME_IN_MS) {
                beginTime = 0;
                return DEFAULT_WAIT_TIME_IN_MS;
            } else {
                return 0;
            }
        }
    }

    public synchronized void addListenedPartition(final DrcClientListener listener, final Partition partition) {
        listeners.add(listener);
        listenedPartitions.add(partition);
        count.incrementAndGet();
    }

    public synchronized void removeListenedPartition(final DrcClientListener listener, final Partition partition) {
        listeners.remove(listener);
        listenedPartitions.remove(partition);
        count.decrementAndGet();
    }

    public void shutdown() throws Exception {
        exited = true;
        this.interrupt();
        this.join();
        logger.info("cluster listener has been shutdown...");
    }

    public int getMessageQueueSize() {
        return msgQueue.size();
    }
}