package com.alibaba.tmq.common.util;

import java.util.*;

import com.alibaba.tmq.client.util.StringUtil;
import com.alibaba.tmq.common.domain.*;
import com.alibaba.tmq.common.domain.result.Result;
import com.alibaba.tmq.common.domain.result.ResultCode;
import com.alibaba.tmq.common.exception.TMQException;
import com.alibaba.tmq.common.proxy.ProxyService;
import com.taobao.eagleeye.EagleEye;

/**
 * Message相关工具
 * @author tianyao.myc
 *
 */
public class MessageUtil {

	/** 代理服务 */
	public static final ProxyService proxyService = new ProxyService();
	
	/**
	 * 检查messageList
	 *  messageUtil
	 *  messageList
	 *
	 */
	public static Result<Boolean> checkMessage(MessageUtil messageUtil, List<? extends Message> messageList) {
		
		if(null == messageList || messageList.size() <= 0) {
			throw new RuntimeException(ResultCode.MESSAGE_LIST_EMPTY_ERROR.getInformation());
		}
		
		Message firstMessage = messageList.get(0);//获取第一个Message
		
		//获取第一个MessageKey
		String messageKey = firstMessage.getMessageKey();
		
		//获取第一个Topic
		String topic = firstMessage.getTopic();
		
		for(Message message : messageList) {
			
			if(! firstMessage.getClass().getSimpleName().equals(message.getClass().getSimpleName())) {
				throw new RuntimeException(ResultCode.MESSAGE_CLASS_NOT_ALL_THE_SAME_ERROR.getInformation());
			}
			
			if(! isCompleteArguments(message)) {
				throw new RuntimeException(ResultCode.MESSAGE_ARGUMENTS_ERROR.getInformation());
			}
			
			if(StringUtil.isBlank(message.getMessageKey())) {
				throw new RuntimeException(ResultCode.MESSAGE_KEY_NULL_ERROR.getInformation());
			}

			if(! message.getMessageKey().equals(messageKey)) {
				throw new RuntimeException(ResultCode.MESSAGE_KEY_NOT_ALL_THE_SAME_ERROR.getInformation());
			}
			
			//检查消息
			Result<Boolean> checkResult = checkMessage(messageUtil, message);
			
			if(! message.getTopic().equals(topic)) {
				throw new RuntimeException(ResultCode.MESSAGE_TOPIC_NOT_ALL_THE_SAME_ERROR.getInformation());
			}
			
			if(! checkResult.getData().booleanValue()) {
				return checkResult;
			}
			
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}
	
	public static boolean fireTimeOut(Date fireTime) {
		
		Date timeLine = TimeUtil.increaseDate(new Date(), Calendar.YEAR, 3);
		
		return ! fireTime.before(timeLine);
	}
	
	/**
	 * 判断是否完整消息参数
	 *  message
	 *
	 */
	public static boolean isCompleteArguments(Message message) {
		
		if((null == message.getFireTime() && StringUtil.isBlank(message.getCronExpression())) 
				|| StringUtil.isBlank(message.getTopic()) 
				|| null == message.getBody()) {
			return false;
		}
		
		return true;
	}
	
	/**
	 * 检查message
	 *  messageUtil
	 *  message
	 *
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	public static Result<Boolean> checkMessage(MessageUtil messageUtil, Message message) {
		
		if(null == message) {
			throw new RuntimeException(ResultCode.MESSAGE_NULL_ERROR.getInformation());
		}
		
		if(! (message instanceof CreateMessage) && ! (message instanceof UpdateMessage) && ! (message instanceof DeleteMessage)) {
			throw new RuntimeException(ResultCode.MESSAGE_CLASS_ERROR.getInformation());
		}
		
		//参数映射表
		Map<String, KeyValuePair<Class<?>[], Object[]>> argumentsTable = message.getArgumentsTable();
		
		Iterator iterator = argumentsTable.entrySet().iterator();
		while (iterator.hasNext()) {
		    Map.Entry entry = (Map.Entry)iterator.next();
		    String field = (String)entry.getKey();
		    KeyValuePair<Class<?>[], Object[]> pair = (KeyValuePair<Class<?>[], Object[]>)entry.getValue();
		    
		    Result<Boolean> result = (Result<Boolean>)proxyService.invokeMethod(messageUtil, field, pair.getKey(), pair.getValue());
		    
		    if(! result.getData().booleanValue()) {
		    	throw new RuntimeException(result.getResultCode().getInformation());
		    }
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}
	
	/**
	 * 检查messageKey
	 *  messageKey
	 *
	 */
	public Result<Boolean> messageKey(String messageKey) {
		
		if(StringUtil.isBlank(messageKey)) {
			throw new RuntimeException(ResultCode.MESSAGE_KEY_NULL_ERROR.getInformation());
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}

	/**
	 * 检查topic
	 *  topic
	 *
	 */
	public Result<Boolean> topic(String topic) {
		
		if(StringUtil.isBlank(topic)) {
			throw new RuntimeException(ResultCode.MESSAGE_TOPIC_NULL_ERROR.getInformation());
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}
	
	/**
	 * 检查unitRoute
	 *  unitRoute
	 *
	 */
	public Result<Boolean> unitRoute(String unitRoute) {
		
		if(StringUtil.isBlank(unitRoute)) {
			throw new RuntimeException(ResultCode.MESSAGE_UNIT_ROUTE_NULL_ERROR.getInformation());
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}
	
	/**
	 * 检查tag
	 *  tag
	 *
	 */
	public Result<Boolean> tag(String tag) {
		
		if(StringUtil.isBlank(tag)) {
			throw new RuntimeException(ResultCode.MESSAGE_TAG_NULL_ERROR.getInformation());
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}
	
	/**
	 * 检查fireTime
	 *  fireTime
	 *
	 */
	public Result<Boolean> fireTime(Date fireTime) {
		
		if(null == fireTime) {
			throw new RuntimeException(ResultCode.MESSAGE_FIRE_TIME_NULL_ERROR.getInformation());
		}
		
		if(fireTimeOut(fireTime)) {
			throw new RuntimeException(ResultCode.MESSAGE_FIRE_TIME_ERROR.getInformation() + ", fireTime:" + TimeUtil.date2SecondsString(fireTime));
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}
	
	/**
	 * 检查body
	 *  body
	 *
	 */
	public Result<Boolean> body(byte[] body) {
		
		if(null == body || body.length <= 0) {
			throw new RuntimeException(ResultCode.MESSAGE_BODY_NULL_ERROR.getInformation());
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}
	
	/**
	 * 检查cronExpression
	 *  message
	 *  cronExpression
	 *
	 */
	public Result<Boolean> cronExpression(Message message, String cronExpression) {
		
		if(StringUtil.isBlank(cronExpression)) {
			throw new RuntimeException(ResultCode.MESSAGE_CRON_EXPRESSION_NULL_ERROR.getInformation());
		}
		
		if(! CronExpressionUtil.isValidExpression(cronExpression)) {
			throw new RuntimeException(ResultCode.MESSAGE_CRON_EXPRESSION_INVALID_ERROR.getInformation());
		}
		
		Date fireTime = CronExpressionUtil.getNextValidTime(cronExpression, new Date());
		if(null == fireTime) {
			throw new RuntimeException(ResultCode.MESSAGE_CRON_EXPRESSION_OBSOLETE_ERROR.getInformation());
		} else {
			message.setFireTime(fireTime);
		}
		
		if(fireTimeOut(fireTime)) {
			throw new RuntimeException(ResultCode.MESSAGE_FIRE_TIME_ERROR.getInformation() + ", fireTime:" + TimeUtil.date2SecondsString(fireTime));
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}
	
	/**
	 * 检查startTime
	 *  startTime
	 *
	 */
	public Result<Boolean> startTime(Date startTime) {
		
		if(null == startTime) {
			throw new RuntimeException(ResultCode.MESSAGE_START_TIME_NULL_ERROR.getInformation());
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}
	
	/**
	 * 检查totalExeCount
	 *  totalExeCount
	 *
	 */
	public Result<Boolean> totalExeCount(long totalExeCount) {
		
		if(totalExeCount < 0L) {
			throw new RuntimeException(ResultCode.MESSAGE_TOTAL_EXE_COUNT_ERROR.getInformation());
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}
	
	/**
	 * 检查endTime
	 *  endTime
	 *
	 */
	public Result<Boolean> endTime(Date endTime) {
		
		if(null == endTime) {
			throw new RuntimeException(ResultCode.MESSAGE_END_TIME_NULL_ERROR.getInformation());
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}
	
	/**
	 * 检查startTime和endTime
	 *  startTime
	 *  endTime
	 *
	 */
	public Result<Boolean> checkStartTimeAndEndTime(Date startTime, Date endTime) {
		
		startTime(startTime);
		
		endTime(endTime);
		
		if(startTime.after(endTime)) {
			throw new RuntimeException(ResultCode.MESSAGE_START_TIME_ERROR.getInformation());
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}
	
	/**
	 * 检查消息
	 *  message
	 *
	 */
	public static Result<Boolean> check4Send(Message message) {
		
		if(null == message) {
			throw new RuntimeException(ResultCode.MESSAGE_NULL_ERROR.getInformation());
		}
		
		if(message instanceof CreateMessage || message instanceof UpdateMessage || message instanceof DeleteMessage) {
			throw new RuntimeException(ResultCode.MESSAGE_OLD_CLASS_ERROR.getInformation());
		}
		
		if(null == message.getFireTime() && StringUtil.isBlank(message.getCronExpression())) {
			throw new RuntimeException(ResultCode.MESSAGE_TIME_BOTH_NULL_ERROR.getInformation());
		}
		
		if(message.getFireTime() != null && fireTimeOut(message.getFireTime())) {
			throw new RuntimeException(ResultCode.MESSAGE_FIRE_TIME_ERROR.getInformation() + ", fireTime:" + TimeUtil.date2SecondsString(message.getFireTime()));
		}
		
		if(StringUtil.isNotBlank(message.getCronExpression())) {
			
			if(! CronExpressionUtil.isValidExpression(message.getCronExpression())) {
				throw new RuntimeException(ResultCode.MESSAGE_CRON_EXPRESSION_INVALID_ERROR.getInformation());
			}
			
			Date fireTime = CronExpressionUtil.getNextValidTime(message.getCronExpression(), new Date());
			if(null == fireTime) {
				throw new RuntimeException(ResultCode.MESSAGE_CRON_EXPRESSION_OBSOLETE_ERROR.getInformation());
			} else {
				message.setFireTime(fireTime);
			}
			
		}
		
		if(StringUtil.isBlank(message.getTopic())) {
			throw new RuntimeException(ResultCode.MESSAGE_TOPIC_NULL_ERROR.getInformation());
		}
		
		if((null == message.getBody() || message.getBody().length <= 0)) {
			throw new RuntimeException(ResultCode.MESSAGE_BODY_NULL_ERROR.getInformation());
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}
	
	/**
	 * 批量检查消息
	 *  messageList
	 *
	 */
	public static Result<Boolean> check4Send(List<Message> messageList) {
		
		if(null == messageList || messageList.size() <= 0) {
			throw new RuntimeException(ResultCode.MESSAGE_LIST_EMPTY_ERROR.getInformation());
		}
		
		//获取第一个MessageKey
		String messageKey = messageList.get(0).getMessageKey();
		
		//获取第一个Topic
		String topic = messageList.get(0).getTopic();
		
		for(Message message : messageList) {
			
			if(StringUtil.isBlank(message.getMessageKey())) {
				throw new RuntimeException(ResultCode.MESSAGE_KEY_NULL_ERROR.getInformation());
			}
			
			if(! message.getMessageKey().equals(messageKey)) {
				throw new RuntimeException(ResultCode.MESSAGE_KEY_NOT_ALL_THE_SAME_ERROR.getInformation());
			}
			
			//检查消息
			Result<Boolean> checkResult = check4Send(message);

			if(! message.getTopic().equals(topic)) {
				throw new RuntimeException(ResultCode.MESSAGE_TOPIC_NOT_ALL_THE_SAME_ERROR.getInformation());
			}
			
			if(! checkResult.getData().booleanValue()) {
				return checkResult;
			}

			//全链路压测标记打入
			if (EagleEye.getUserData("t")!=null){
				if(EagleEye.getUserData("t").equals("1")){
					message.setTb_eagleeyex_t("1");
				}
			}

		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}
	
	/**
	 * 渲染消息
	 *  producerId
	 *  message
	 *  status
	 *  messageType
	 *
	 *  TMQException
	 */
	public static KeyValuePair<String, Message> rendering(
			String producerId, Message message, MessageStatus status, MessageType messageType) throws TMQException {
		
		String messageId = IdAndKeyUtil.acquireUniqueId();//生成消息唯一ID
		
		//业务全局唯一Key
		String messageKey = IdAndKeyUtil.acquireUniqueKey(messageId, message);
		
		if(StringUtil.isNotBlank(producerId)) {
			message.setProducerId(producerId);
		}
		
		if(status != null && ! (message instanceof DeleteMessage)) {
			message.setStatus(status.getStatus());//设置消息状态
		}
		
		if(message instanceof CreateMessage || message instanceof UpdateMessage) {
			message.setMessageId(messageId);
		}
		
		message.setMessageKey(messageKey);
		
		if(messageType != null) {
			message.setType(messageType.getType());
		}
		
		return new KeyValuePair<String, Message>(messageId, message);
	}
	
	/**
	 * 渲染消息列表
	 *  producerId
	 *  messageList
	 *  status
	 *  messageType
	 *
	 *  TMQException
	 */
	public static List<KeyValuePair<String, Message>> rendering(
			String producerId, List<? extends Message> messageList, MessageStatus status, MessageType messageType) throws TMQException {
		
		List<KeyValuePair<String, Message>> messageKeyValueList = new ArrayList<KeyValuePair<String, Message>>();
		
		for(Message message : messageList) {
			
			//渲染消息
			messageKeyValueList.add(rendering(producerId, message, status, messageType));
		}
		
		return messageKeyValueList;
	}
	
	/**
	 * 渲染消息
	 *  producerId
	 *  message
	 *  status
	 *  messageType
	 *
	 *  TMQException
	 */
	public static KeyValuePair<String, Message> renderingMessage(
			String producerId, Message message, MessageStatus status, MessageType messageType) throws TMQException {
		
		String messageId = IdAndKeyUtil.acquireUniqueId();//生成消息唯一ID
		
		//业务全局唯一Key
		String messageKey = IdAndKeyUtil.acquireUniqueKey(messageId, message);
		
		if(StringUtil.isNotBlank(producerId)) {
			message.setProducerId(producerId);
		}
		
		if(status != null) {
			message.setStatus(status.getStatus());//设置消息状态
		}
		
		message.setMessageId(messageId);
		message.setMessageKey(messageKey);
		
		if(messageType != null) {
			message.setType(messageType.getType());
		}
		
		return new KeyValuePair<String, Message>(messageId, message);
	}
	
	/**
	 * 渲染消息列表
	 *  producerId
	 *  messageList
	 *  status
	 *  messageType
	 *
	 *  TMQException
	 */
	public static List<KeyValuePair<String, Message>> renderingMessage(
			String producerId, List<Message> messageList, MessageStatus status, MessageType messageType) throws TMQException {
		
		List<KeyValuePair<String, Message>> messageKeyValueList = new ArrayList<KeyValuePair<String, Message>>();
		
		for(Message message : messageList) {
			
			//渲染消息
			messageKeyValueList.add(renderingMessage(producerId, message, status, messageType));
		}
		
		return messageKeyValueList;
	}
	
	/**
	 * 设置消息状态
	 *  message
	 *  status
	 */
	public static void setMessageStatus(Message message, MessageStatus status) {
		
		//设置状态
		message.setStatus(status.getStatus());
	}
	
	/**
	 * 批量设置消息状态
	 *  messageList
	 *  status
	 */
	public static void setMessageStatus(List<Message> messageList, MessageStatus status) {
		
		for(Message message : messageList) {
			
			//设置消息状态
			setMessageStatus(message, status);
		}
		
	}
	

	
	/**
	 * 检查消息
	 *  message
	 *
	 */
	public static Result<Boolean> check4Update(Message message) {
		
		if(null == message) {
			throw new RuntimeException(ResultCode.MESSAGE_NULL_ERROR.getInformation());
		}
		
		if(message instanceof CreateMessage || message instanceof UpdateMessage || message instanceof DeleteMessage) {
			throw new RuntimeException(ResultCode.MESSAGE_OLD_CLASS_ERROR.getInformation());
		}
		
		if(StringUtil.isBlank(message.getMessageId()) && StringUtil.isBlank(message.getMessageKey())) {
			throw new RuntimeException(ResultCode.MESSAGE_ID_AND_KEY_NULL_ERROR.getInformation());
		}
		
		if(StringUtil.isBlank(message.getTopic())) {
			throw new RuntimeException(ResultCode.MESSAGE_TOPIC_NULL_ERROR.getInformation());
		}
		
		if(message.getFireTime() != null && fireTimeOut(message.getFireTime())) {
			throw new RuntimeException(ResultCode.MESSAGE_FIRE_TIME_ERROR.getInformation() + ", fireTime:" + TimeUtil.date2SecondsString(message.getFireTime()));
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}
	
	/**
	 * 检查消息
	 *  message
	 *
	 */
	public static Result<Boolean> check4UpdateSingle(Message message) {
		
		if(null == message) {
			throw new RuntimeException(ResultCode.MESSAGE_NULL_ERROR.getInformation());
		}
		
		if(message instanceof CreateMessage || message instanceof UpdateMessage || message instanceof DeleteMessage) {
			throw new RuntimeException(ResultCode.MESSAGE_OLD_CLASS_ERROR.getInformation());
		}
		
		if(StringUtil.isBlank(message.getMessageId()) && StringUtil.isBlank(message.getMessageKey())) {
			throw new RuntimeException(ResultCode.MESSAGE_ID_AND_KEY_NULL_ERROR.getInformation());
		}
		
		if(null == message.getFireTime() && (null == message.getBody() || message.getBody().length <= 0)) {
			throw new RuntimeException(ResultCode.MESSAGE_FIRE_TIME_AND_BODY_NULL_ERROR.getInformation());
		}
		
		if(message.getFireTime() != null && fireTimeOut(message.getFireTime())) {
			throw new RuntimeException(ResultCode.MESSAGE_FIRE_TIME_ERROR.getInformation() + ", fireTime:" + TimeUtil.date2SecondsString(message.getFireTime()));
		}
		
		if(StringUtil.isBlank(message.getTopic())) {
			throw new RuntimeException(ResultCode.MESSAGE_TOPIC_NULL_ERROR.getInformation());
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}
	
	/**
	 * 批量检查消息
	 *  messageList
	 *
	 */
	public static Result<Boolean> check4Update(List<Message> messageList) {
		
		if(null == messageList || messageList.size() <= 0) {
			throw new RuntimeException(ResultCode.MESSAGE_LIST_EMPTY_ERROR.getInformation());
		}
		
		//获取第一个MessageKey
		String messageKey = messageList.get(0).getMessageKey();
		
		//获取第一个Topic
		String topic = messageList.get(0).getTopic();
		
		for(Message message : messageList) {
			
			if(StringUtil.isBlank(message.getMessageKey())) {
				throw new RuntimeException(ResultCode.MESSAGE_KEY_NULL_ERROR.getInformation());
			}
			
			if(! message.getMessageKey().equals(messageKey)) {
				throw new RuntimeException(ResultCode.MESSAGE_KEY_NOT_ALL_THE_SAME_ERROR.getInformation());
			}
			
			//检查消息
			Result<Boolean> checkResult = check4Update(message);
			
			if(! message.getTopic().equals(topic)) {
				throw new RuntimeException(ResultCode.MESSAGE_TOPIC_NOT_ALL_THE_SAME_ERROR.getInformation());
			}
			
			if(! checkResult.getData().booleanValue()) {
				return checkResult;
			}
			
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}
	
	/**
	 * 检查消息
	 *  message
	 *
	 */
	public static Result<Boolean> check4Get(Message message) {
		
		if(null == message) {
			throw new RuntimeException(ResultCode.MESSAGE_NULL_ERROR.getInformation());
		}
		
		if(message instanceof CreateMessage || message instanceof UpdateMessage || message instanceof DeleteMessage) {
			throw new RuntimeException(ResultCode.MESSAGE_OLD_CLASS_ERROR.getInformation());
		}
		
		if(StringUtil.isBlank(message.getTopic())) {
			throw new RuntimeException(ResultCode.MESSAGE_TOPIC_NULL_ERROR.getInformation());
		}
		
		if(StringUtil.isBlank(message.getMessageKey())) {
			throw new RuntimeException(ResultCode.MESSAGE_KEY_NULL_ERROR.getInformation());
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}

	
	/**
	 * 检查消息
	 *  message
	 *
	 */
	public static Result<Boolean> check4Delete(Message message) {
		
		if(null == message) {
			throw new RuntimeException(ResultCode.MESSAGE_NULL_ERROR.getInformation());
		}
		
		if(message instanceof CreateMessage || message instanceof UpdateMessage || message instanceof DeleteMessage) {
			throw new RuntimeException(ResultCode.MESSAGE_OLD_CLASS_ERROR.getInformation());
		}
		
		if(StringUtil.isBlank(message.getMessageId()) && StringUtil.isBlank(message.getMessageKey())) {
			throw new RuntimeException(ResultCode.MESSAGE_ID_AND_KEY_NULL_ERROR.getInformation());
		}
		
		if(StringUtil.isBlank(message.getTopic())) {
			throw new RuntimeException(ResultCode.MESSAGE_TOPIC_NULL_ERROR.getInformation());
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}
	
	/**
	 * 批量检查消息
	 *  messageList
	 *
	 */
	public static Result<Boolean> check4Delete(List<Message> messageList) {
		
		if(null == messageList || messageList.size() <= 0) {
			throw new RuntimeException(ResultCode.MESSAGE_LIST_EMPTY_ERROR.getInformation());
		}
		
		//获取第一个MessageKey
		String messageKey = messageList.get(0).getMessageKey();
		
		for(Message message : messageList) {
			
			if(StringUtil.isBlank(message.getMessageKey())) {
				throw new RuntimeException(ResultCode.MESSAGE_KEY_NULL_ERROR.getInformation());
			}
			
			if(! message.getMessageKey().equals(messageKey)) {
				throw new RuntimeException(ResultCode.MESSAGE_KEY_NOT_ALL_THE_SAME_ERROR.getInformation());
			}
			
			//检查消息
			Result<Boolean> checkResult = check4Delete(message);
			
			if(! checkResult.getData().booleanValue()) {
				return checkResult;
			}
			
		}
		
		return new Result<Boolean>(true, ResultCode.SUCCESS);
	}
	
	/**
	 * 重置参数
	 *  message
	 *  clusterId
	 */
	public static void reset(Message message, int clusterId){
		
		if(null == message) {
			throw new RuntimeException("message is null");
		}
		
		message.setRetryConsumers(null);
		message.setRetryCount(0);
		message.setClusterId(clusterId);
	}

	/**
	 * 批量重置参数
	 *  messageList
	 *  clusterId
	 */
	public static void reset(List<? extends Message> messageList, int clusterId){
		
		if(null == messageList || messageList.size() <= 0) {
			throw new RuntimeException("messageList is empty");
		}
		
		for(Message message : messageList) {
			
			//重置参数
			reset(message, clusterId);
			
		}
		
	}
	
	/**
	 * 时间精确到秒
	 *  messageList
	 */
	public static void date2Seconds(List<Message> messageList) {
		
		if(null == messageList || messageList.size() <= 0) {
			throw new RuntimeException("messageList is empty");
		}
		
		for(Message message : messageList) {
			
			//时间精确到秒
			date2Seconds(message);
		}
		
	}
	
	/**
	 * 时间精确到秒
	 *  message
	 */
	public static void date2Seconds(Message message) {
		
		if(null == message.getFireTime()) {
			return ;
		}
		
		message.setFireTime(TimeUtil.date2Seconds(message.getFireTime()));
	}
	
	/**
	 * 重试复制消息
	 *  message
	 *
	 */
	public static Message retryClone(Message message) {
		
		if(null == message) {
			return null;
		}
		
		Message retryMessage = new Message();
		retryMessage.setId(0L);
		retryMessage.setGmtCreate(null);
		retryMessage.setGmtModified(null);
		retryMessage.setMessageId(IdAndKeyUtil.acquireUniqueId());
		retryMessage.setFireTime(new Date());
		retryMessage.setProducerId(message.getProducerId());
		retryMessage.setTopic(message.getTopic());
		retryMessage.setTag(message.getTag());
		retryMessage.setBody(message.getBody());
		retryMessage.setRetryConsumers(null);
		retryMessage.setStatus(MessageStatus.RETRY.getStatus());
		retryMessage.setRetryCount(0);
		retryMessage.setMessageKey(message.getMessageKey());
		retryMessage.setType(MessageType.NORMAL_CRON_RETRY.getType());
		retryMessage.setClusterId(message.getClusterId());
		retryMessage.setUnitRoute(message.getUnitRoute());
		retryMessage.setCronExpression(null);
		retryMessage.setStartTime(null);
		retryMessage.setTotalExeCount(0L);
		retryMessage.setCurrentExeCount(message.getCurrentExeCount());
		retryMessage.setEndTime(null);
		retryMessage.setFireTimeMinute(null);
		
		return retryMessage;
	}
}
