package com.alibaba.tmq.common.monitor;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.commons.logging.Log;

import com.alibaba.tmq.common.domain.KeyValuePair;
import com.alibaba.tmq.common.domain.remoting.Pair;
import com.alibaba.tmq.common.exception.InitException;
import com.alibaba.tmq.common.monitor.callback.Display;
import com.alibaba.tmq.common.monitor.timer.MethodCountTimer;

/**
 * 方法监控
 * @author tianyao.myc
 *
 */
public class MethodMonitor {

	private final Log logger;

	/** 方法统计映射表 */
	private final ConcurrentHashMap<String, Pair<AtomicLong/** 执行次数 */, AtomicLong/** 执行耗时 */>> methodCountTable =
			new ConcurrentHashMap<String, Pair<AtomicLong, AtomicLong>>();

	//监控表
	private final ConcurrentSkipListMap<String, ConcurrentSkipListMap<String, KeyValuePair<AtomicLong, AtomicLong>>> monitorTable =
			new ConcurrentSkipListMap<String, ConcurrentSkipListMap<String, KeyValuePair<AtomicLong, AtomicLong>>>();

	//展现信息列表
	private final List<Display> displayList = new ArrayList<Display>();

	/** 定时调度服务 */
	private ScheduledExecutorService executorService = Executors
			.newScheduledThreadPool(1, new ThreadFactory() {

				int index = 0;

				public Thread newThread(Runnable runnable) {

					index ++;

					return new Thread(runnable, "TMQ-MethodMonitor-Thread-" + index);
				}

			});

	public MethodMonitor(Log logger) {
		this.logger = logger;
	}


	public void init() throws InitException {

		/** 初始化方法统计定时器 */
//		initMethodCountTimer();

	}


	public void initMethodCountTimer() throws InitException {
		try {
			executorService.scheduleAtFixedRate(new MethodCountTimer(logger, this.displayList, this.methodCountTable),
					0L, 1000L, TimeUnit.MILLISECONDS);
		} catch (Throwable e) {
			throw new InitException("[MethodMonitor]: initMethodCountTimer error", e);
		}
	}


	public void addDisplay(Display display) {
		this.displayList.add(display);
	}


	public void methodCount(String key, long startTime) {
		try {
			Pair<AtomicLong, AtomicLong> countPair = getCountPair(key);
			/** 次数累加 */
			AtomicLong counter = countPair.getObject1();
			counter.incrementAndGet();
			/** 耗时累加 */
			AtomicLong totalTime = countPair.getObject2();
			totalTime.addAndGet(System.currentTimeMillis() - startTime);
		} catch (Throwable e) {
			logger.error("[MethodMonitor]: methodCount error, key:" + key + ", startTime:" + startTime);
		}
	}


	public void methodCount(String roleKey, String methodKey, long startTime) {

		try {

			ConcurrentSkipListMap<String, KeyValuePair<AtomicLong, AtomicLong>> counterTable = getCounterTable(roleKey);
			KeyValuePair<AtomicLong, AtomicLong> counterPair = getCounterPair(counterTable, methodKey);

			/** 次数累加 */
			AtomicLong counter = counterPair.getKey();
			counter.incrementAndGet();

			/** 耗时累加 */
			AtomicLong totalTime = counterPair.getValue();
			totalTime.addAndGet(System.currentTimeMillis() - startTime);
		} catch (Throwable e) {
			logger.error("[MethodMonitor]: methodCount error"
					+ ", roleKey:" + roleKey
					+ ", methodKey:" + methodKey
					+ ", startTime:" + startTime);
		}
	}


	private ConcurrentSkipListMap<String, KeyValuePair<AtomicLong, AtomicLong>> getCounterTable(String roleKey) {

		ConcurrentSkipListMap<String, KeyValuePair<AtomicLong, AtomicLong>> counterTable = this.monitorTable.get(roleKey);
		if(null == counterTable) {
			counterTable = new ConcurrentSkipListMap<String, KeyValuePair<AtomicLong, AtomicLong>>();
			ConcurrentSkipListMap<String, KeyValuePair<AtomicLong, AtomicLong>> existCounterTable = this.monitorTable.putIfAbsent(roleKey, counterTable);
			if(existCounterTable != null) {
				counterTable = existCounterTable;
			}
		}

		return counterTable;
	}


	private KeyValuePair<AtomicLong, AtomicLong> getCounterPair(
			ConcurrentSkipListMap<String, KeyValuePair<AtomicLong, AtomicLong>> counterTable, String methodKey) {

		KeyValuePair<AtomicLong, AtomicLong> counterPair = counterTable.get(methodKey);
		if(null == counterPair) {
			counterPair = new KeyValuePair<AtomicLong, AtomicLong>(new AtomicLong(0L), new AtomicLong(0L));
			KeyValuePair<AtomicLong, AtomicLong> existCounterPair = counterTable.putIfAbsent(methodKey, counterPair);
			if(existCounterPair != null) {
				counterPair = existCounterPair;
			}
		}

		return counterPair;
	}

	private Pair<AtomicLong, AtomicLong> getCountPair(String key) {
		Pair<AtomicLong, AtomicLong> countPair = this.methodCountTable.get(key);
		if(null == countPair) {
			countPair = new Pair<AtomicLong, AtomicLong>(new AtomicLong(0L), new AtomicLong(0L));
			Pair<AtomicLong, AtomicLong> existCountPair = this.methodCountTable.putIfAbsent(key, countPair);
			if(existCountPair != null) {
				countPair = existCountPair;
			}
		}
		return countPair;
	}

	public List<Display> getDisplayList() {
		return displayList;
	}

	public ConcurrentSkipListMap<String, ConcurrentSkipListMap<String, KeyValuePair<AtomicLong, AtomicLong>>> getMonitorTable() {
		return monitorTable;
	}
	
}
