package com.alibaba.tmq.client.system.producer.implement;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.alibaba.tmq.client.context.ClientContext;
import com.alibaba.tmq.client.system.producer.Producer;
import com.alibaba.tmq.client.system.producer.config.ProducerConfig;
import com.alibaba.tmq.client.system.producer.executer.ProducerExecuter;
import com.alibaba.tmq.client.util.StringUtil;
import com.alibaba.tmq.common.constants.Constants;
import com.alibaba.tmq.common.context.InvocationContext;
import com.alibaba.tmq.common.domain.Message;
import com.alibaba.tmq.common.domain.remoting.ConnectionChannel;
import com.alibaba.tmq.common.domain.result.DeleteResult;
import com.alibaba.tmq.common.domain.result.Result;
import com.alibaba.tmq.common.domain.result.ResultCode;
import com.alibaba.tmq.common.domain.result.UpdateResult;
import com.alibaba.tmq.common.exception.TMQClientException;
import com.alibaba.tmq.common.exception.TMQException;
import com.alibaba.tmq.common.exception.TMQServerException;
import com.alibaba.tmq.common.service.ServerService;
import com.alibaba.tmq.common.util.IdAndKeyUtil;
import com.alibaba.tmq.common.util.ListUtil;
import com.alibaba.tmq.common.util.LoggerUtil;
import com.alibaba.tmq.common.util.MessageUtil;
import com.taobao.common.fulllinkstresstesting.SplitEnvUtil;

/**
 * 默认生产者
 * @author tianyao.myc
 *
 */
public class DefaultProducer extends ClientContext implements Producer, Constants {

	private static final Log logger = LogFactory.getLog(DefaultProducer.class);
	
	/** 服务端基础服务 */
	private final ServerService serverService = clientRemoting.proxyInterface(ServerService.class);
	
	/** 各项参数配置 */
	private final ProducerConfig producerConfig;
	
	private final ProducerExecuter producerExecuter;
	
	public DefaultProducer(ProducerConfig producerConfig, ProducerExecuter producerExecuter) {
		this.producerConfig = producerConfig;
		this.producerExecuter = producerExecuter;
	}
	
	/**
	 * 启动
	 */
	public void start() {
		
		if( ! SplitEnvUtil.needStartDTS()){
			logger.error("[DefaultProducer]: start error, because of isolation environment");
			return ;
		}
		
		/** 初始化客户端 */
		try {
			initClient();
		} catch (Throwable e) {
			throw new RuntimeException("[DefaultProducer]: start initClient error"
					+ ", clientConfig:" + clientConfig 
					+ ", producerConfig:" + producerConfig, e);
		}
		
		/** 初始化连接 */
		try {
			clientRemoting.initConnection(producerConfig.getInstanceName(), 
					ROLE_TYPE_PRODUCER, this.producerConfig.getProducerId(), NULL, NULL);
		} catch (Throwable e) {
			logger.error("[DefaultProducer]: start initConnection error"
					+ ", clientConfig:" + clientConfig 
					+ ", producerConfig:" + producerConfig, e);
			
			throw new RuntimeException("[DefaultProducer]: start initConnection error"
					+ ", clientConfig:" + clientConfig 
					+ ", producerConfig:" + producerConfig, e);
		}
		
		//设置Producer状态为启动
		this.producerExecuter.setStart(true);
	}

	/**
	 * 获取消息
	 */
	@Override
	public List<Message> get(Message message) throws TMQException {

		if(! this.producerExecuter.isStart()) {
			throw new TMQClientException("producer has just create, not start");
		}

		try {
			//重置参数，防止用户设置参数导致系统运行错误
			if(StringUtil.isNotBlank(clientConfig.getBackupDomainName())) {
				MessageUtil.reset(message, clientConfig.getBackupClusterId());
			} else {
				MessageUtil.reset(message, clientConfig.getClusterId());
			}
		} catch (Throwable e) {
			throw new TMQClientException(e);
		}

		//检查消息是否正确
		Result<Boolean> checkResult = null;
		try {
			checkResult = MessageUtil.check4Get(message);
		} catch (Throwable e) {
			throw new TMQClientException(e);
		}

		if(! checkResult.getData().booleanValue()) {
			return new ArrayList<Message>();
		}

		//渲染MessageKey
		IdAndKeyUtil.renderingMessageKey(this.producerConfig.getProducerId(), message);

		/** 获取该消息要触发的那台Server */
		String server = clientRemoting.acquireFireServer(message.getMessageKey());
		if(StringUtil.isBlank(server)) {
			throw new TMQServerException(ResultCode.SERVER_ERROR.getInformation());
		}

		//发送获取消息
		List<Message> messageList = get(message, server);

		//如果发送到触发机器失败就往别的机器发送直到尝试完所有机器
		if(null == messageList) {

			List<String> serverList = clientRemoting.getServerList();

			if(StringUtil.isNotBlank(clientConfig.getBackupDomainName())) {
				serverList = clientRemoting.getbackupServerList();
			}

			if(ListUtil.isEmpty(serverList)) {
				throw new TMQServerException(ResultCode.SERVER_ERROR.getInformation());
			}

			//循环尝试其他服务器
			for(String otherServer : serverList) {

				//发送获取消息到其他server
				messageList = get(message, otherServer);
				if(messageList != null) {
					break ;//如果有返回结果就退出循环
				}
			}
		}

		if(null == messageList) {
			throw new TMQClientException(ResultCode.TIMEOUT.getInformation());
		}

		return messageList;
	}

	/**
	 * 获取消息
	 *  message
	 *  server
	 *
	 */
	private List<Message> get(Message message, String server) {
		
		List<Message> messageList = null;
		try {
			InvocationContext.setConnectionChannel(new ConnectionChannel(server, ROLE_TYPE_PRODUCER, this.producerConfig.getProducerId()));
			messageList = serverService.get(message);
		} catch (Throwable e) {
			logger.error("[DefaultProducer]: get message error, message:" + message, e);
		}
		
		return messageList;
	}


	/**
	 * 更新消息
	 */
	public UpdateResult update(Message message) throws TMQException {
		
		if(! this.producerExecuter.isStart()) {
			throw new TMQClientException("producer has just create, not start");
		}

		long startTime = System.currentTimeMillis();
		
		try {
			//重置参数，防止用户设置参数导致系统运行错误
			if(StringUtil.isNotBlank(clientConfig.getBackupDomainName())) {
				MessageUtil.reset(message, clientConfig.getBackupClusterId());
			} else {
				MessageUtil.reset(message, clientConfig.getClusterId());
			}
		} catch (Throwable e) {
			throw new TMQClientException(e);
		}
		
		//检查消息是否正确
		Result<Boolean> checkResult = null;
		try {
			checkResult = MessageUtil.check4UpdateSingle(message);
		} catch (Throwable e) {
			throw new TMQClientException(e);
		}
		
		if(! checkResult.getData().booleanValue()) {
			
			LoggerUtil.info(message, "[TMQ-CLIENT-PRODUCER]-update-single-check-failed", startTime
					, clientConfig.getLocalAddress());//消息跟踪埋点
			
			return new UpdateResult(checkResult);
		}
		
		//渲染MessageKey
		IdAndKeyUtil.renderingMessageKey(this.producerConfig.getProducerId(), message);
		
		/** 获取该消息要触发的那台Server */
		String server = clientRemoting.acquireFireServer(message.getMessageKey());
		if(StringUtil.isBlank(server)) {
			
			LoggerUtil.info(message, "[TMQ-CLIENT-PRODUCER]-update-single-server-failed", startTime
					, clientConfig.getLocalAddress());//消息跟踪埋点
			
			throw new TMQServerException(ResultCode.SERVER_ERROR.getInformation());
		}

		//发送更新消息
		Result<Boolean> updateResult = update(message, server);
		
		//如果发送到触发机器失败就往别的机器发送直到尝试完所有机器
		if(null == updateResult) {
			
			List<String> serverList = clientRemoting.getServerList();
			
			if(StringUtil.isNotBlank(clientConfig.getBackupDomainName())) {
				serverList = clientRemoting.getbackupServerList();
			}
			
			if(ListUtil.isEmpty(serverList)) {
				
				LoggerUtil.info(message, "[TMQ-CLIENT-PRODUCER]-update-single-server-empty", startTime
						, clientConfig.getLocalAddress());//消息跟踪埋点
				
				throw new TMQServerException(ResultCode.SERVER_ERROR.getInformation());
			}
			
			//循环尝试其他服务器
			for(String otherServer : serverList) {
				
				//发送更新消息到其他server
				updateResult = update(message, otherServer);
				if(updateResult != null) {
					break ;//如果有返回结果就退出循环
				}
			}
		}
		
		if(null == updateResult) {
			
			LoggerUtil.info(message, "[TMQ-CLIENT-PRODUCER]-update-single-result-failed", startTime
					, clientConfig.getLocalAddress());//消息跟踪埋点
			
			throw new TMQClientException(ResultCode.TIMEOUT.getInformation());
		}
		
		LoggerUtil.info(message, "[TMQ-CLIENT-PRODUCER]-update-single-" + updateResult, startTime
				, clientConfig.getLocalAddress());//消息跟踪埋点
		
		if(ResultCode.CONNECTION_PRODUCER_IS_NULL_ERROR.equals(updateResult.getResultCode())) {
			throw new TMQServerException(ResultCode.CONNECTION_PRODUCER_IS_NULL_ERROR.getInformation());
		}
		
		return new UpdateResult(updateResult);
	}
	
	/**
	 * 更新消息到server
	 *  message
	 *  server
	 *
	 */
	private Result<Boolean> update(Message message, String server) {
		
		Result<Boolean> updateResult = null;
		try {
			InvocationContext.setConnectionChannel(new ConnectionChannel(server, ROLE_TYPE_PRODUCER, this.producerConfig.getProducerId()));
			updateResult = serverService.update(message);
		} catch (Throwable e) {
			logger.error("[DefaultProducer]: update message error, message:" + message, e);
		}
		
		return updateResult;
	}

	/**
	 * 批量更新消息
	 *  messageList
	 *
	 *  TMQException
	 */
	public UpdateResult updateMessageList(List<Message> messageList) throws TMQException {
		
		long startTime = System.currentTimeMillis();
		
		/** 获取该消息要触发的那台Server */
		String server = clientRemoting.acquireFireServer(messageList.get(0).getMessageKey());
		if(StringUtil.isBlank(server)) {
			
			LoggerUtil.info(messageList, "[TMQ-CLIENT-PRODUCER]-update-mulit-server-failed", startTime
					, clientConfig.getLocalAddress());//消息跟踪埋点
			
			throw new TMQServerException(ResultCode.SERVER_ERROR.getInformation());
		}
		
		//发送更新消息
		Result<Boolean> updateResult = update(messageList, server);

		//如果发送到触发机器失败就往别的机器发送直到尝试完所有机器
		if(null == updateResult) {
			
			List<String> serverList = clientRemoting.getServerList();
			
			if(StringUtil.isNotBlank(clientConfig.getBackupDomainName())) {
				serverList = clientRemoting.getbackupServerList();
			}
			
			if(ListUtil.isEmpty(serverList)) {
				
				LoggerUtil.info(messageList, "[TMQ-CLIENT-PRODUCER]-update-mulit-server-empty", startTime
						, clientConfig.getLocalAddress());//消息跟踪埋点
				
				throw new TMQServerException(ResultCode.SERVER_ERROR.getInformation());
			}
			
			//循环尝试其他服务器
			for(String otherServer : serverList) {
				
				//发送更新消息到其他server
				updateResult = update(messageList, otherServer);
				if(updateResult != null) {
					break ;//如果有返回结果就退出循环
				}
			}
		}
		
		if(null == updateResult) {
			throw new TMQClientException(ResultCode.TIMEOUT.getInformation());
		}
		
		return new UpdateResult(updateResult);
	}

	/**
	 * 更新批量消息到server
	 *  messageList
	 *  server
	 *
	 */
	private Result<Boolean> update(List<Message> messageList, String server) {
		
		Result<Boolean> updateResult = null;
		try {
			InvocationContext.setConnectionChannel(new ConnectionChannel(server, ROLE_TYPE_PRODUCER, this.producerConfig.getProducerId()));
			updateResult = serverService.update(messageList);
		} catch (Throwable e) {
			logger.error("[DefaultProducer]: update messageList error, messageList:" + messageList, e);
		}
		
		return updateResult;
	}
	
	/**
	 * 删除消息
	 */
	public DeleteResult delete(Message message) throws TMQException {
		
		if(! this.producerExecuter.isStart()) {
			throw new TMQClientException("producer has just create, not start");
		}
		
		long startTime = System.currentTimeMillis();
		
		try {
			//重置参数，防止用户设置参数导致系统运行错误
			if(StringUtil.isNotBlank(clientConfig.getBackupDomainName())) {
				MessageUtil.reset(message, clientConfig.getBackupClusterId());
			} else {
				MessageUtil.reset(message, clientConfig.getClusterId());
			}
		} catch (Throwable e) {
			throw new TMQClientException(e);
		}
		
		//检查消息是否正确
		Result<Boolean> checkResult = null;
		try {
			checkResult = MessageUtil.check4Delete(message);
		} catch (Throwable e) {
			throw new TMQClientException(e);
		}
		
		if(! checkResult.getData().booleanValue()) {
			
			LoggerUtil.info(message, "[TMQ-CLIENT-PRODUCER]-delete-single-check-failed", startTime
					, clientConfig.getLocalAddress());//消息跟踪埋点
			
			return new DeleteResult(checkResult);
		}
		
		//渲染MessageKey
		IdAndKeyUtil.renderingMessageKey(this.producerConfig.getProducerId(), message);
		
		/** 获取该消息要触发的那台Server */
		String server = clientRemoting.acquireFireServer(message.getMessageKey());
		if(StringUtil.isBlank(server)) {
			
			LoggerUtil.info(message, "[TMQ-CLIENT-PRODUCER]-delete-single-server-failed", startTime
					, clientConfig.getLocalAddress());//消息跟踪埋点
			
			throw new TMQServerException(ResultCode.SERVER_ERROR.getInformation());
		}
		
		//发送删除消息
		Result<Boolean> deleteResult = delete(message, server);

		//如果发送到触发机器失败就往别的机器发送直到尝试完所有机器
		if(null == deleteResult) {
			
			List<String> serverList = clientRemoting.getServerList();
			
			if(StringUtil.isNotBlank(clientConfig.getBackupDomainName())) {
				serverList = clientRemoting.getbackupServerList();
			}
			
			if(ListUtil.isEmpty(serverList)) {
				
				LoggerUtil.info(message, "[TMQ-CLIENT-PRODUCER]-delete-single-server-empty", startTime
						, clientConfig.getLocalAddress());//消息跟踪埋点
				
				throw new TMQServerException(ResultCode.SERVER_ERROR.getInformation());
			}
		
			//循环尝试其他服务器
			for(String otherServer : serverList) {
				
				//发送删除消息到其他server
				deleteResult = delete(message, otherServer);
				if(deleteResult != null) {
					break ;//如果有返回结果就退出循环
				}
			}
		}
		
		if(null == deleteResult) {
			
			LoggerUtil.info(message, "[TMQ-CLIENT-PRODUCER]-delete-single-result-failed", startTime
					, clientConfig.getLocalAddress());//消息跟踪埋点
			
			throw new TMQClientException(ResultCode.TIMEOUT.getInformation());
		}
		
		LoggerUtil.info(message, "[TMQ-CLIENT-PRODUCER]-delete-single-" + deleteResult, startTime
				, clientConfig.getLocalAddress());//消息跟踪埋点
		
		if(ResultCode.CONNECTION_PRODUCER_IS_NULL_ERROR.equals(deleteResult.getResultCode())) {
			throw new TMQServerException(ResultCode.CONNECTION_PRODUCER_IS_NULL_ERROR.getInformation());
		}
		
		return new DeleteResult(deleteResult);
	}

	/**
	 * 删除消息
	 *  message
	 *  server
	 *
	 */
	private Result<Boolean> delete(Message message, String server) {
		
		Result<Boolean> deleteResult = null;
		try {
			InvocationContext.setConnectionChannel(new ConnectionChannel(server, ROLE_TYPE_PRODUCER, this.producerConfig.getProducerId()));
			deleteResult = serverService.delete(message);
		} catch (Throwable e) {
			logger.error("[DefaultProducer]: delete message error, message:" + message, e);
		}
		
		return deleteResult;
	}
	
	/**
	 * 批量删除消息
	 *  messageList
	 *
	 *  TMQException
	 */
	public DeleteResult delete(List<Message> messageList) throws TMQException {

		//重置参数，防止用户设置参数导致系统运行错误
		if(StringUtil.isNotBlank(clientConfig.getBackupDomainName())) {
			MessageUtil.reset(messageList, clientConfig.getBackupClusterId());
		} else {
			MessageUtil.reset(messageList, clientConfig.getClusterId());
		}

		//检查消息是否正确
		Result<Boolean> checkResult = MessageUtil.check4Delete(messageList);
		if(! checkResult.getData().booleanValue()) {
			return new DeleteResult(checkResult);
		}
		
		//渲染MessageKey
		IdAndKeyUtil.renderingMessageKey(this.producerConfig.getProducerId(), messageList);
		
		/** 获取该消息要触发的那台Server */
		String server = clientRemoting.acquireFireServer(messageList.get(0).getMessageKey());
		if(StringUtil.isBlank(server)) {
			return new DeleteResult(false, ResultCode.SERVER_ERROR);
		}
		
		//发送删除消息
		Result<Boolean> deleteResult = null;
		try {
			InvocationContext.setConnectionChannel(new ConnectionChannel(server, ROLE_TYPE_PRODUCER, this.producerConfig.getProducerId()));
			deleteResult = serverService.delete(messageList);
		} catch (Throwable e) {
			logger.error("[DefaultProducer]: delete message error, messageList:" + messageList, e);
		}
		
		if(null == deleteResult) {
			return new DeleteResult(false, ResultCode.DELETE_MESSAGE_TIMEOUT);//如果返回值是null那就是超时了
		}
		
		return new DeleteResult(deleteResult);
	}

	/**
	 * 关闭
	 */
	public void shutdown() {
		
	}

}
