package com.alibaba.tmq.client.service;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.alibaba.tmq.client.TMQFactory;
import com.alibaba.tmq.client.context.ClientContext;
import com.alibaba.tmq.client.system.consumer.executer.ConsumerExecuter;
import com.alibaba.tmq.client.system.producer.executer.TransactionProducerExecuter;
import com.alibaba.tmq.common.context.InvocationContext;
import com.alibaba.tmq.common.domain.*;
import com.alibaba.tmq.common.domain.remoting.ConnectionChannel;
import com.alibaba.tmq.common.domain.result.Result;
import com.alibaba.tmq.common.domain.result.ResultCode;
import com.alibaba.tmq.common.service.ClientService;
import com.alibaba.tmq.common.util.LoggerUtil;

/**
 * 客户端通用基础服务
 * @author tianyao.myc
 *
 */
public class DefaultClientService extends ClientContext implements ClientService {

	private static final Log logger = LogFactory.getLog(DefaultClientService.class);
	
	/**
	 * 心跳检查
	 *
	 */
	public Result<String> heartBeatCheck() {
		
		ConnectionChannel connectionChannel = InvocationContext.acquireConnectionChannel();
		
		try {
			
			//心跳计数器递增
			clientRemoting.increaseCounter(connectionChannel);
		} catch (Throwable e) {
			
			logger.error("[DefaultClientService]: heartBeatCheck error, connectionChannel:" + connectionChannel, e);
			
			return new Result<String>("something wrong with connectionChannel:" + connectionChannel, ResultCode.FAILURE);
		}
		
		return new Result<String>("I am alive !", ResultCode.SUCCESS);
	}

	/**
	 * 推送消息
	 *  consumerId
	 *  message
	 *
	 */
	public Result<Action> push(String consumerId, Message message) {
		
		logger.error("[DefaultClientService]: push error, consumerId:" + consumerId + ", message:" + message);
		
		return new Result<Action>(Action.RETRY, ResultCode.FAILURE);
	}
	
	/**
	 * 推送消息
	 */
	public Result<Action> push(ConsumerKey consumerKey, Message message) {

		ConnectionChannel connectionChannel = InvocationContext.acquireConnectionChannel();
		
		long startTime = System.currentTimeMillis();
		
		ConsumerExecuter consumerExecuter = TMQFactory.aquireConsumerExecuter(consumerKey, connectionChannel.getInstanceName());
		
		if(null == consumerExecuter) {
			
			logger.error("[DefaultClientService]: push consumer is null error"
					+ ", consumerKey:" + consumerKey + ", message:" + message 
					+ ", instanceName:" + connectionChannel.getInstanceName() );
			
			LoggerUtil.info(message, "[TMQ-CLIENT-CONSUMER]-push-consumerExecuter-failed-" + consumerKey + "-" + connectionChannel.getInstanceName(), 
					startTime, clientConfig.getLocalAddress());//消息跟踪埋点
			
			return new Result<Action>(Action.RETRY, ResultCode.CONSUMER_NULL_ERROR);
		}
		
		Result<Action> consumeResult = null;
		try {
			consumeResult = consumerExecuter.consume(consumerKey, message, connectionChannel);
		} catch (Throwable e) {
			
			logger.error("[DefaultClientService]: consume error"
					+ ", consumerKey:" + consumerKey 
					+ ", instanceName:" + connectionChannel.getInstanceName() 
					+ ", message:" + message, e);
			
			LoggerUtil.info(message, "[TMQ-CLIENT-CONSUMER]-push-consume-error-" + consumerKey + "-" + connectionChannel.getInstanceName(), 
					startTime, clientConfig.getLocalAddress());//消息跟踪埋点
			
			return new Result<Action>(Action.RETRY, ResultCode.CONSUME_EXCEPTION);
		}
		
		if(null == consumeResult) {
			
			LoggerUtil.info(message, "[TMQ-CLIENT-CONSUMER]-push-result-failed-" + consumerKey + "-" + connectionChannel.getInstanceName(), 
					startTime, clientConfig.getLocalAddress());//消息跟踪埋点
			
			return new Result<Action>(Action.RETRY, ResultCode.FAILURE);
		}
		
		LoggerUtil.info(message, "[TMQ-CLIENT-CONSUMER]-push-" + consumerKey + "-" + connectionChannel.getInstanceName() + "-" + consumeResult, 
				startTime, clientConfig.getLocalAddress());//消息跟踪埋点
		
		return consumeResult;
	}

	/**
	 * 事务消息检查
	 *  producerKey
	 *  message
	 *
	 */
	public Result<TransactionStatus> check(ProducerKey producerKey, Message message) {
		TransactionProducerExecuter producerExecuter = TMQFactory.aquireTransactionProducerExecuter(producerKey.getProducerId(), null);
		
		if(null == producerExecuter) {
			
			logger.error("[DefaultClientService]: check producer is null error"
					+ ", producerId:" + producerKey.getProducerId() + ", message:" + message);
			
			return new Result<TransactionStatus>(TransactionStatus.UnKnow, ResultCode.PRODUCER_NULL_ERROR);
		}
		
		Result<TransactionStatus> checkResult = null;
		try {
			checkResult = producerExecuter.check(message);
		} catch (Throwable e) {
			
			logger.error("[DefaultClientService]: check error"
					+ ", producerId:" + producerKey.getProducerId() + ", message:" + message);
			
			return new Result<TransactionStatus>(TransactionStatus.UnKnow, ResultCode.CHECK_EXCEPTION);
		}
		
		if(null == checkResult) {
			return new Result<TransactionStatus>(TransactionStatus.UnKnow, ResultCode.FAILURE);
		}
		
		return checkResult;
	}
	
}
