package com.alibaba.tmq.client;

import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.alibaba.tmq.client.context.ClientContext;
import com.alibaba.tmq.client.system.consumer.Consumer;
import com.alibaba.tmq.client.system.consumer.config.ConsumerConfig;
import com.alibaba.tmq.client.system.consumer.executer.ConsumerExecuter;
import com.alibaba.tmq.client.system.producer.NormalProducer;
import com.alibaba.tmq.client.system.producer.config.ProducerConfig;
import com.alibaba.tmq.client.system.producer.executer.NormalProducerExecuter;
import com.alibaba.tmq.client.system.producer.executer.TransactionProducerExecuter;
import com.alibaba.tmq.client.util.StringUtil;
import com.alibaba.tmq.common.constants.PropertyKeyConstants;
import com.alibaba.tmq.common.domain.ConsumerKey;
import com.alibaba.tmq.common.util.RandomUtil;

/**
 * TMQ工厂
 * @author tianyao.myc
 *
 */
public class TMQFactory extends ClientContext {

	private static final Log logger = LogFactory.getLog(TMQFactory.class); 
	
	/** 普通Producer映射表 */
	private static final ConcurrentHashMap<String, ConcurrentHashMap<String, NormalProducerExecuter>> normalProducerTable = 
			new ConcurrentHashMap<String, ConcurrentHashMap<String, NormalProducerExecuter>>();
	
	/** 事务Producer映射表 */
	private static final ConcurrentHashMap<String, ConcurrentHashMap<String, TransactionProducerExecuter>> transactionProducerTable = 
			new ConcurrentHashMap<String, ConcurrentHashMap<String, TransactionProducerExecuter>>();
	
	/** Consumer映射表 */
	private static final ConcurrentHashMap<String, ConcurrentHashMap<String, ConsumerExecuter>> consumerTable = 
			new ConcurrentHashMap<String, ConcurrentHashMap<String, ConsumerExecuter>>();
	
	/**
	 * 创建普通生产者
	 *  properties
	 *
	 */
	public synchronized static NormalProducer createNormalProducer(Properties properties) {
		
		if(null == properties) {
			throw new RuntimeException("[TMQFactory]: createNormalProducer error, properties is null");
		}
		
		/** 设置各项参数配置 */
		ProducerConfig producerConfig = new ProducerConfig();
		producerConfig.setProducerId(properties.getProperty(PropertyKeyConstants.ProducerId));
		
		if(StringUtil.isBlank(producerConfig.getProducerId())) {
			throw new RuntimeException("[TMQFactory]: createNormalProducer error, ProducerId is empty");
		}
		
		String clusterId = properties.getProperty(PropertyKeyConstants.ClusterId);
		if(StringUtil.isNotBlank(clusterId)) {
			producerConfig.setClusterId(Integer.parseInt(clusterId));
			clientConfig.setClusterId(producerConfig.getClusterId());
		}
		
		String backupClusterId = properties.getProperty(PropertyKeyConstants.BackupClusterId);
		if(StringUtil.isNotBlank(backupClusterId)) {
			clientConfig.setBackupClusterId(Integer.parseInt(backupClusterId));
		}
		
		String instanceName = properties.getProperty(PropertyKeyConstants.InstanceName);
		if(StringUtil.isNotBlank(instanceName)) {
			producerConfig.setInstanceName(instanceName);
		}
		
		ConcurrentHashMap<String, NormalProducerExecuter> producerMap = normalProducerTable.get(producerConfig.getProducerId());
		if(null == producerMap) {
			producerMap = new ConcurrentHashMap<String, NormalProducerExecuter>();
			normalProducerTable.put(producerConfig.getProducerId(), producerMap);
		}
		
		NormalProducerExecuter producerExecuter = producerMap.get(producerConfig.getInstanceName());
		if(null == producerExecuter) {
			producerExecuter = new NormalProducerExecuter(producerConfig);
			producerMap.put(producerConfig.getInstanceName(), producerExecuter);
		} else {
			throw new RuntimeException("[TMQFactory]: createNormalProducer error"
					+ ", producerId and instanceName is already exists"
					+ ", producerId:" + producerConfig.getProducerId() 
					+ ", instanceName:" + producerConfig.getInstanceName());
		}
		
		return producerExecuter.getProducer();
	}
	
//	/**
//	 * 创建事务生产者
//	 *  properties
//	 *  localTransactionChecker
//	 *
//	 */
//	public synchronized static TransactionProducer createTransactionProducer(Properties properties, LocalTransactionChecker localTransactionChecker) {
//		
//		if(null == properties) {
//			throw new RuntimeException("[TMQFactory]: createTransactionProducer error, properties is null");
//		}
//		
//		//如果localTransactionChecker为空就抛出异常
//		if(null == localTransactionChecker) {
//			throw new RuntimeException("[TMQFactory]: createTransactionProducer error, localTransactionChecker is null");
//		}
//		
//		/** 设置各项参数配置 */
//		ProducerConfig producerConfig = new ProducerConfig();
//		producerConfig.setProducerId(properties.getProperty(PropertyKeyConstants.ProducerId));
//		
//		if(StringUtil.isBlank(producerConfig.getProducerId())) {
//			throw new RuntimeException("[TMQFactory]: createTransactionProducer error, ProducerId is empty");
//		}
//		
//		String clusterId = properties.getProperty(PropertyKeyConstants.ClusterId);
//		if(StringUtil.isNotBlank(clusterId)) {
//			producerConfig.setClusterId(Integer.parseInt(clusterId));
//			clientConfig.setClusterId(producerConfig.getClusterId());
//		}
//		
//		String instanceName = properties.getProperty(PropertyKeyConstants.InstanceName);
//		if(StringUtil.isNotBlank(instanceName)) {
//			producerConfig.setInstanceName(instanceName);
//		}
//		
//		ConcurrentHashMap<String, TransactionProducerExecuter> producerMap = transactionProducerTable.get(producerConfig.getProducerId());
//		if(null == producerMap) {
//			producerMap = new ConcurrentHashMap<String, TransactionProducerExecuter>();
//			transactionProducerTable.put(producerConfig.getProducerId(), producerMap);
//		}
//		
//		TransactionProducerExecuter producerExecuter = producerMap.get(producerConfig.getInstanceName());
//		if(null == producerExecuter) {
//			producerExecuter = new TransactionProducerExecuter(producerConfig, localTransactionChecker);
//			producerMap.put(producerConfig.getInstanceName(), producerExecuter);
//		} else {
//			throw new RuntimeException("[TMQFactory]: createTransactionProducer error"
//					+ ", producerId and instanceName is already exists"
//					+ ", producerId:" + producerConfig.getProducerId() 
//					+ ", instanceName:" + producerConfig.getInstanceName());
//		}
//		
//		return producerExecuter.getProducer();
//	}
	
	/**
	 * 创建消费者
	 *  properties
	 *
	 */
	public synchronized static Consumer createConsumer(Properties properties) {
		
		if(null == properties) {
			throw new RuntimeException("[TMQFactory]: createConsumer error, properties is null");
		}
		
		/** 设置各项参数配置 */
		ConsumerConfig consumerConfig = new ConsumerConfig();
		consumerConfig.setConsumerId(properties.getProperty(PropertyKeyConstants.ConsumerId));
		
		if(StringUtil.isBlank(consumerConfig.getConsumerId())) {
			throw new RuntimeException("[TMQFactory]: createConsumer error, ConsumerId is empty");
		}

		String clusterId = properties.getProperty(PropertyKeyConstants.ClusterId);
		if(StringUtil.isNotBlank(clusterId)) {
			consumerConfig.setClusterId(Integer.parseInt(clusterId));
			clientConfig.setClusterId(consumerConfig.getClusterId());
		}
		
		String backupClusterId = properties.getProperty(PropertyKeyConstants.BackupClusterId);
		if(StringUtil.isNotBlank(backupClusterId)) {
			clientConfig.setBackupClusterId(Integer.parseInt(backupClusterId));
		}
		
		String instanceName = properties.getProperty(PropertyKeyConstants.InstanceName);
		if(StringUtil.isNotBlank(instanceName)) {
			consumerConfig.setInstanceName(instanceName);
		}
		
		ConcurrentHashMap<String, ConsumerExecuter> consumerMap = consumerTable.get(consumerConfig.getConsumerId());
		if(null == consumerMap) {
			consumerMap = new ConcurrentHashMap<String, ConsumerExecuter>();
			consumerTable.put(consumerConfig.getConsumerId(), consumerMap);
		}
		
		ConsumerExecuter consumerExecuter = consumerMap.get(consumerConfig.getInstanceName());
		if(null == consumerExecuter) {
			consumerExecuter = new ConsumerExecuter(consumerConfig, consumerConfig.getInstanceName());
			consumerMap.put(consumerConfig.getInstanceName(), consumerExecuter);
		} else {
			throw new RuntimeException("[TMQFactory]: createConsumer error"
					+ ", consumerId and instanceName is already exists"
					+ ", consumerId:" + consumerConfig.getConsumerId() 
					+ ", instanceName:" + consumerConfig.getInstanceName());
		}
		
		return consumerExecuter.getConsumer();
	}
	
	/**
	 * 移除普通生产者
	 *  producerId
	 *  instanceName
	 */
	public static void removeNormalProducer(String producerId, String instanceName) {
		
		ConcurrentHashMap<String, NormalProducerExecuter> producerMap = normalProducerTable.get(producerId);
		if(null == producerMap) {
			return ;
		}
		
		if(StringUtil.isBlank(instanceName)) {
			try {
				normalProducerTable.remove(producerId);
			} catch (Throwable e) {
				logger.error("[TMQFactory]: removeNormalProducer error, producerId:" + producerId, e);
			}
		} else {
			try {
				producerMap.remove(instanceName);
			} catch (Throwable e) {
				logger.error("[TMQFactory]: removeNormalProducer error"
						+ ", producerId:" + producerId 
						+ ", instanceName:" + instanceName, e);
			}
		}
		
	}
	
	/**
	 * 移除事务生产者
	 *  producerId
	 *  instanceName
	 */
	public static void removeTransactionProducer(String producerId, String instanceName) {
		
		ConcurrentHashMap<String, TransactionProducerExecuter> producerMap = transactionProducerTable.get(producerId);
		if(null == producerMap) {
			return ;
		}
		
		if(StringUtil.isBlank(instanceName)) {
			try {
				transactionProducerTable.remove(producerId);
			} catch (Throwable e) {
				logger.error("[TMQFactory]: removeTransactionProducer error, producerId:" + producerId, e);
			}
		} else {
			try {
				producerMap.remove(instanceName);
			} catch (Throwable e) {
				logger.error("[TMQFactory]: removeTransactionProducer error"
						+ ", producerId:" + producerId 
						+ ", instanceName:" + instanceName, e);
			}
		}
	}
	
	/**
	 * 获取事务生产者执行器
	 *  producerId
	 *  instanceName
	 *
	 */
	public static TransactionProducerExecuter aquireTransactionProducerExecuter(String producerId, String instanceName) {
		
		ConcurrentHashMap<String, TransactionProducerExecuter> producerMap = transactionProducerTable.get(producerId);
		if(null == producerMap || producerMap.size() <= 0) {
			return null;
		}
		
		if(StringUtil.isBlank(instanceName)) {
			
			return RandomUtil.getRandomObject4Map(producerMap);
			
		} else {
			return producerMap.get(instanceName);
		}
		
	}
	
	/**
	 * 获取消费者执行器
	 *  consumerKey
	 *  instanceName
	 *
	 */
	public static ConsumerExecuter aquireConsumerExecuter(ConsumerKey consumerKey, String instanceName) {
		
		ConcurrentHashMap<String, ConsumerExecuter> consumerMap = consumerTable.get(consumerKey.getConsumerId());
		if(null == consumerMap || consumerMap.size() <= 0) {
			return null;
		}
		
		if(StringUtil.isBlank(instanceName)) {
			
			return RandomUtil.getRandomObject4Map(consumerMap);
			
		} else {
			return consumerMap.get(instanceName);
		}
		
	}
	
	/**
	 * 移除消费者
	 *  consumerId
	 *  instanceName
	 */
	public static void removeConsumer(String consumerId, String instanceName) {
		
		ConcurrentHashMap<String, ConsumerExecuter> consumerMap = consumerTable.get(consumerId);
		if(null == consumerMap) {
			return ;
		}
		
		String beforeConsumerMap = consumerMap.toString();
		
		if(StringUtil.isBlank(instanceName)) {
			try {
				consumerTable.remove(consumerId);
			} catch (Throwable e) {
				logger.error("[TMQFactory]: removeConsumer error, consumerId:" + consumerId, e);
			}
		} else {
			try {
				consumerMap.remove(instanceName);
			} catch (Throwable e) {
				logger.error("[TMQFactory]: removeConsumer error"
						+ ", consumerId:" + consumerId 
						+ ", instanceName:" + instanceName, e);
			}
		}
		

		logger.warn("[TMQFactory]: removeConsumer"
				+ ", consumerId:" + consumerId 
				+ ", instanceName:" + instanceName 
				+ ", beforeConsumerMap:" + beforeConsumerMap 
				+ ", afterConsumerMap" + consumerMap);
		
	}

	public static ConcurrentHashMap<String, ConcurrentHashMap<String, NormalProducerExecuter>> getNormalproducertable() {
		return normalProducerTable;
	}

	public static ConcurrentHashMap<String, ConcurrentHashMap<String, TransactionProducerExecuter>> getTransactionproducertable() {
		return transactionProducerTable;
	}

	public static ConcurrentHashMap<String, ConcurrentHashMap<String, ConsumerExecuter>> getConsumertable() {
		return consumerTable;
	}

}
