package com.alibaba.dts.sdk;



import com.alibaba.dts.common.constants.Constants;
import com.alibaba.dts.common.domain.result.Result;
import com.alibaba.dts.common.domain.result.ResultCode;
import com.alibaba.dts.common.domain.store.Job;
import com.alibaba.dts.common.domain.store.assemble.*;
import com.alibaba.dts.common.logger.SchedulerXLoggerFactory;
import com.alibaba.dts.common.logger.innerlog.Logger;
import com.alibaba.dts.common.remoting.protocol.RemotingSerializable;
import com.alibaba.dts.common.service.HttpService;
import com.alibaba.dts.common.util.CheckUtil;
import com.alibaba.dts.common.util.DiamondHelper;
import com.alibaba.dts.common.util.GroupIdUtil;
import com.alibaba.dts.common.util.StringUtil;
import com.alibaba.dts.sdk.client.DtsHttpClient;
import com.alibaba.dts.sdk.context.SDKContext;
import com.alibaba.dts.sdk.util.exception.SDKModeUnsupportException;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;

/**
 * Created by luliang on 14/12/24.
 */
public class DtsSecureCommonSDKManager implements DtsSecureSDKManager {

    private static final Logger logger = SchedulerXLoggerFactory.getLogger(DtsSecureCommonSDKManager.class);

    private SDKMode mode = SDKMode.ALIYUN_MODE;

    public long clusterId;

    private DtsHttpClient client = new DtsHttpClient();

    public DtsSecureCommonSDKManager(SDKMode mode) {
        this.mode = mode;
        if (mode == SDKMode.DAILY_MODE) {
            client.setDomainUrl(SDKContext.DTS_DOMAIN_DAILY_URL);
        } else if (mode == SDKMode.ONLINE_MODE) {
            client.setDomainUrl(SDKContext.DTS_DOMAIN_ONLINE_URL);
        } else if (mode == SDKMode.ALIYUN_MODE) {
            client.setDomainUrl(SDKContext.DTS_DOMAIN_ALIYUN_URL);
        } else if (mode == SDKMode.USA_MODE) {
            client.setDomainUrl(SDKContext.DTS_DOMAIN_USA_URL);
        } else {
            throw new SDKModeUnsupportException("sdk mode not support!");
        }
    }

    public DtsSecureCommonSDKManager(String url) {
        if (StringUtil.isBlank(url)) {
            throw new RuntimeException("url不能为空!");
        }
        client.setDomainUrl(url);
    }

    public DtsSecureCommonSDKManager() {

        String domainName = null;
        try {
            domainName = DiamondHelper.getData(HttpService.DOMAIN_NAME_DATA_ID, 10 * 1000L);
            client.setDomainUrl("http://" + domainName + "/dts-console");
            DiamondHelper.addListener(HttpService.DOMAIN_NAME_DATA_ID, new DiamondHelper.DataListener() {
                @Override
                public void receiveConfigInfo(String dataId, String configInfo) {
                    client.setDomainUrl("http://" + configInfo + "/dts-console");
                }
            });
        } catch (Throwable e) {
            throw new RuntimeException("[DtsCommonSDKManager]: get domainName from diamond error", e);
        }

        if (StringUtil.isBlank(domainName)) {
            throw new RuntimeException("[DtsCommonSDKManager]: domainName is blank error");
        }

    }


    public Result<Long> createJob(String userGroupId, Job job) {
        Result<Long> result = new Result<Long>();
        // 检查userGroupId
        if (StringUtil.isBlank(userGroupId) || !GroupIdUtil.checkClientGroupId(userGroupId)) {
            result.setResultCode(ResultCode.USER_PARAMETER_ERROR);
            result.getResultCode().setInformation("组名不正确!");
            return result;
        }
        Result<Boolean> checkResult = CheckUtil.checkUserConfigJob(job);
        if (!checkResult.getData()) {
            result.setResultCode(checkResult.getResultCode());
            result.getResultCode().setInformation(checkResult.getResultCode().getInformation());
            return result;
        }

        client.setTarget("sdkManager.do").setSubmitAction("event_submit_do_create_job");
        client.addParameter("userGroupId", userGroupId);
        client.addParameter("jobProcessor", job.getJobProcessor())
                .addParameter("jobType", String.valueOf(job.getType()))
                .addParameter("cronExpression", job.getCronExpression())
                .addParameter("jobDesc", StringUtil.isEmpty(job.getDescription()) ? StringUtil.EMPTY_STRING : job.getDescription())
                .addParameter("firePolicy", String.valueOf(job.getMaxInstanceAmount()))
                .addParameter("jobArguments", StringUtil.isEmpty(job.getJobArguments()) ? StringUtil.EMPTY_STRING : job.getJobArguments())
                .addParameter("jobStatus", String.valueOf(job.getStatus()))
                .addParameter("clusterCode", String.valueOf(job.getClusterCode()));

        String postJson = client.doPost();

        if (StringUtil.isBlank(postJson)) {
            logger.error("[DtsCommonSDKManager]: createJob json isBlank"
                    + ", userGroupId:" + userGroupId
                    + ", job:" + job
                    + ", postJson:" + postJson);
            result.setResultCode(ResultCode.FAILURE);
            return result;
        }

        try {
            JSONObject jsonResult = JSON.parseObject(postJson);
            Boolean success = (Boolean) jsonResult.get(Constants.SUCCESS);
            if (success == false) {
                result.setResultCode(ResultCode.FAILURE);
                result.getResultCode().setInformation((String) jsonResult.get(Constants.ERROR_MSG));
            } else {
                result.setResultCode(ResultCode.SUCCESS);
                result.setData(jsonResult.getLong("jobId"));
            }

        } catch (Throwable e) {
            logger.error("[DtsCommonSDKManager]: createJob"
                    + ", userGroupId:" + userGroupId
                    + ", job:" + job
                    + ", postJson:" + postJson, e);
            result.setResultCode(ResultCode.SDK_IO_ERROR);
        }
        return result;
    }

    public Result<Integer> deleteJob(String userGroupId,long jobId) {
        Result<Integer> result = new Result<Integer>();
        client.setTarget("sdkManager.do").setSubmitAction("event_submit_do_delete_job");
        client.addParameter("jobId", String.valueOf(jobId));

        String postJson = client.doPost();

        if (StringUtil.isBlank(postJson)) {
            logger.error("[DtsCommonSDKManager]: deleteJob json isBlank"
                    + ", jobId:" + jobId
                    + ", postJson:" + postJson);
            result.setResultCode(ResultCode.FAILURE);
            return result;
        }

        try {
            JSONObject jsonResult = JSON.parseObject(postJson);
            Boolean success = (Boolean) jsonResult.get(Constants.SUCCESS);
            if (success == false) {
                result.setResultCode(ResultCode.FAILURE);
                result.getResultCode().setInformation((String) jsonResult.get(Constants.ERROR_MSG));
            } else {
                result.setResultCode(ResultCode.SUCCESS);
                result.setData(jsonResult.getInteger("deleteCount"));
            }
        } catch (Throwable e) {
            logger.error("[DtsCommonSDKManager]: deleteJob error"
                    + ", jobId:" + jobId
                    + ", postJson:" + postJson, e);
            result.setResultCode(ResultCode.SDK_IO_ERROR);
        }
        return result;
    }


    public Result<Job> getJobConfig(String userGroupId,long jobId) {
        Result<Job> result = new Result<Job>();
        client.setTarget("sdkManager.do").setSubmitAction("event_submit_do_get_job_config");
        client.addParameter("jobId", String.valueOf(jobId));

        String postJson = client.doPost();

        if (StringUtil.isBlank(postJson)) {
            logger.error("[DtsCommonSDKManager]: getJobConfig json isBlank"
                    + ", jobId:" + jobId
                    + ", postJson:" + postJson);
            result.setResultCode(ResultCode.FAILURE);
            return result;
        }

        try {
            JSONObject jsonResult = JSON.parseObject(postJson);
            Boolean success = (Boolean) jsonResult.get(Constants.SUCCESS);
            if (success == false) {
                result.setResultCode(ResultCode.FAILURE);
                result.getResultCode().setInformation((String) jsonResult.get(Constants.ERROR_MSG));
            } else {
                result.setResultCode(ResultCode.SUCCESS);
                Job job = RemotingSerializable.fromJson(jsonResult.get("jobConfig").toString(), Job.class);
                result.setData(job);
            }
        } catch (Throwable e) {
            logger.error("[DtsCommonSDKManager]: getJobConfig error"
                    + ", jobId:" + jobId
                    + ", postJson:" + postJson, e);
            result.setResultCode(ResultCode.SDK_IO_ERROR);
        }
        return result;
    }


    public Result<Integer> updateJob(String groupId, Job job) {
        Result<Integer> result = new Result<Integer>();
        Result<Boolean> checkResult = CheckUtil.checkUserConfigJob(job);
        if (!checkResult.getData()) {
            result.setResultCode(checkResult.getResultCode());
            result.getResultCode().setInformation(checkResult.getResultCode().getInformation());
            return result;
        }
        client.setTarget("sdkManager.do").setSubmitAction("event_submit_do_update_job");
        client.addParameter("jobId", String.valueOf(job.getId()))
                .addParameter("jobProcessor", job.getJobProcessor())
                .addParameter("jobType", String.valueOf(job.getType()))
                .addParameter("cronExpression", job.getCronExpression())
                .addParameter("jobDesc", StringUtil.isEmpty(job.getDescription()) ? StringUtil.EMPTY_STRING : job.getDescription())
                .addParameter("firePolicy", String.valueOf(job.getMaxInstanceAmount()))
                .addParameter("jobArguments", StringUtil.isEmpty(job.getJobArguments()) ? StringUtil.EMPTY_STRING : job.getJobArguments())
                .addParameter("groupId", groupId);

        String postJson = client.doPost();

        if (StringUtil.isBlank(postJson)) {
            logger.error("[DtsCommonSDKManager]: updateJob json isBlank"
                    + ", groupId:" + groupId
                    + ", job:" + job
                    + ", postJson:" + postJson);
            result.setResultCode(ResultCode.FAILURE);
            return result;
        }

        try {
            JSONObject jsonResult = JSON.parseObject(postJson);
            Boolean success = (Boolean) jsonResult.get(Constants.SUCCESS);
            if (success == false) {
                result.setResultCode(ResultCode.FAILURE);
                result.getResultCode().setInformation((String) jsonResult.get(Constants.ERROR_MSG));
            } else {
                result.setResultCode(ResultCode.SUCCESS);
                result.setData(jsonResult.getInteger("updateCount"));
            }
        } catch (Throwable e) {
            logger.error("[DtsCommonSDKManager]: updateJob error"
                    + ", groupId:" + groupId
                    + ", job:" + job
                    + ", postJson:" + postJson, e);
            result.setResultCode(ResultCode.SDK_IO_ERROR);
        }
        return result;
    }


    public Result<Boolean> instanceRunJob(String userGroupId, long jobId) {
        Result<Boolean> result = new Result<Boolean>();
        client.setTarget("sdkManager.do").setSubmitAction("event_submit_do_instance_start_job");
        client.addParameter("jobId", String.valueOf(jobId)).addParameter("groupId", userGroupId);

        String postJson = client.doPost();

        if (StringUtil.isBlank(postJson)) {
            logger.error("[DtsCommonSDKManager]: instanceRunJob json isBlank"
                    + ", userGroupId:" + userGroupId
                    + ", jobId:" + jobId
                    + ", postJson:" + postJson);
            result.setResultCode(ResultCode.FAILURE);
            return result;
        }

        try {
            JSONObject jsonResult = JSON.parseObject(postJson);
            Boolean success = (Boolean) jsonResult.get(Constants.SUCCESS);
            if (success == false) {
                result.setResultCode(ResultCode.FAILURE);
                result.getResultCode().setInformation(postJson);
            } else {
                result.setResultCode(ResultCode.SUCCESS);
                result.getResultCode().setInformation(postJson);
                result.setData(true);
            }
        } catch (Throwable e) {
            logger.error("[DtsCommonSDKManager]: instanceRunJob error"
                    + ", userGroupId:" + userGroupId
                    + ", jobId:" + jobId
                    + ", postJson:" + postJson, e);
            result.setResultCode(ResultCode.SDK_IO_ERROR);
        }
        return result;
    }


    public Result<JobStatus> getJobRunningStatus(String userGroupId,long jobId) {
        Result<JobStatus> result = new Result<JobStatus>();
        client.setTarget("sdkManager.do").setSubmitAction("event_submit_do_get_job_status");
        client.addParameter("jobId", String.valueOf(jobId));

        String postJson = client.doPost();

        if (StringUtil.isBlank(postJson)) {
            logger.error("[DtsCommonSDKManager]: getJobRunningStatus json isBlank"
                    + ", jobId:" + jobId
                    + ", postJson:" + postJson);
            result.setResultCode(ResultCode.FAILURE);
            return result;
        }

        try {
            JSONObject jsonResult = JSON.parseObject(postJson);
            Boolean success = (Boolean) jsonResult.get(Constants.SUCCESS);
            if (success == false) {
                result.setResultCode(ResultCode.FAILURE);
                result.getResultCode().setInformation((String) jsonResult.get(Constants.ERROR_MSG));
            } else {
                result.setResultCode(ResultCode.SUCCESS);
                JobStatus jobStatus = RemotingSerializable.fromJson(jsonResult.get("status").toString(), JobStatus.class);
                result.setData(jobStatus);
            }
        } catch (Throwable e) {
            logger.error("[DtsCommonSDKManager]: getJobRunningStatus error"
                    + ", jobId:" + jobId
                    + ", postJson:" + postJson, e);
            result.setResultCode(ResultCode.SDK_IO_ERROR);
        }
        return result;
    }


    public Result<Boolean> instanceStopJob(String userGroupId,long jobId) {
        Result<Boolean> result = new Result<Boolean>();
        client.setTarget("sdkManager.do").setSubmitAction("event_submit_do_instance_stop_job");
        client.addParameter("jobId", String.valueOf(jobId));

        String postJson = client.doPost();

        if (StringUtil.isBlank(postJson)) {
            logger.error("[DtsCommonSDKManager]: instanceStopJob json isBlank"
                    + ", jobId:" + jobId
                    + ", postJson:" + postJson);
            result.setResultCode(ResultCode.FAILURE);
            return result;
        }

        try {
            JSONObject jsonResult = JSON.parseObject(postJson);
            Boolean success = (Boolean) jsonResult.get(Constants.SUCCESS);
            if (success == false) {
                result.setResultCode(ResultCode.FAILURE);
                result.getResultCode().setInformation((String) jsonResult.get(Constants.ERROR_MSG));
            } else {
                result.setResultCode(ResultCode.SUCCESS);
                result.setData(true);
            }
        } catch (Throwable e) {
            logger.error("[DtsCommonSDKManager]: instanceStopJob error"
                    + ", jobId:" + jobId
                    + ", postJson:" + postJson, e);
            result.setResultCode(ResultCode.SDK_IO_ERROR);
        }
        return result;
    }


    public Result<Boolean> disableJob(String userGroupId,long jobId) {
        Result<Boolean> result = new Result<Boolean>();
        client.setTarget("sdkManager.do").setSubmitAction("event_submit_do_disable_job");
        client.addParameter("jobId", String.valueOf(jobId));

        String postJson = client.doPost();

        if (StringUtil.isBlank(postJson)) {
            logger.error("[DtsCommonSDKManager]: disableJob json isBlank"
                    + ", jobId:" + jobId
                    + ", postJson:" + postJson);
            result.setResultCode(ResultCode.FAILURE);
            return result;
        }

        try {
            JSONObject jsonResult = JSON.parseObject(postJson);
            Boolean success = (Boolean) jsonResult.get(Constants.SUCCESS);
            if (success == false) {
                result.setResultCode(ResultCode.FAILURE);
                result.getResultCode().setInformation((String) jsonResult.get(Constants.ERROR_MSG));
            } else {
                result.setResultCode(ResultCode.SUCCESS);
                result.setData(true);
            }
        } catch (Throwable e) {
            logger.error("[DtsCommonSDKManager]: disableJob error"
                    + ", jobId:" + jobId
                    + ", postJson:" + postJson, e);
            result.setResultCode(ResultCode.SDK_IO_ERROR);
        }
        return result;
    }

    public Result<Boolean> enableJob(String userGroupId,long jobId) {
        Result<Boolean> result = new Result<Boolean>();
        client.setTarget("sdkManager.do").setSubmitAction("event_submit_do_enable_job");
        client.addParameter("jobId", String.valueOf(jobId));

        String postJson = client.doPost();

        if (StringUtil.isBlank(postJson)) {
            logger.error("[DtsCommonSDKManager]: enableJob json isBlank"
                    + ", jobId:" + jobId
                    + ", postJson:" + postJson);
            result.setResultCode(ResultCode.FAILURE);
            return result;
        }

        try {
            JSONObject jsonResult = JSON.parseObject(postJson);
            Boolean success = (Boolean) jsonResult.get(Constants.SUCCESS);
            if (success == false) {
                result.setResultCode(ResultCode.FAILURE);
                result.getResultCode().setInformation((String) jsonResult.get(Constants.ERROR_MSG));
            } else {
                result.setResultCode(ResultCode.SUCCESS);
                result.setData(true);
            }
        } catch (Throwable e) {
            logger.error("[DtsCommonSDKManager]: enableJob error"
                    + ", jobId:" + jobId
                    + ", postJson:" + postJson, e);
            result.setResultCode(ResultCode.SDK_IO_ERROR);
        }
        return result;
    }


    public long getClusterId() {
        return clusterId;
    }

    public void setClusterId(long clusterId) {
        this.clusterId = clusterId;
    }

    public SDKMode getMode() {
        return mode;
    }

    public void setMode(SDKMode mode) {
        this.mode = mode;
    }
}
