package com.alibaba.dts.client.executor.simple;

import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.springframework.util.CollectionUtils;

import com.alibaba.dts.client.executor.job.context.ClientContextImpl;
import com.alibaba.dts.client.executor.simple.processor.SimpleTaskProcessor;
import com.alibaba.dts.client.service.JobRunningStateManager;
import com.alibaba.dts.common.domain.ExecutableTask;
import com.alibaba.dts.common.domain.result.Result;
import com.alibaba.dts.common.domain.result.ResultCode;
import com.alibaba.dts.common.domain.store.Job;
import com.alibaba.dts.common.domain.store.JobInstanceSnapshot;
import com.alibaba.dts.common.logger.SchedulerXLoggerFactory;
import com.alibaba.dts.common.logger.innerlog.Logger;
import com.taobao.eagleeye.EagleEye;

/**
 * 简单job执行池
 * @author tianyao.myc
 *
 */
public class SimplePool {

	private static final Logger logger = SchedulerXLoggerFactory.getLogger(SimplePool.class);

	/** 简单Job映射表 */
	private ConcurrentHashMap<Long, ConcurrentHashMap<Long, SimpleTaskProcessor>> simpleJobTable =
			new ConcurrentHashMap<Long, ConcurrentHashMap<Long, SimpleTaskProcessor>>();

	private final ClientContextImpl clientContext;

	public SimplePool(final ClientContextImpl clientContext) {
		this.clientContext = clientContext;
	}

	public void stopService(){

		Iterator iterator = simpleJobTable.entrySet().iterator();

		long jobCount = 0;
		long processorCount = 0;

		while(iterator.hasNext()) {

			long jobid=0;
			jobCount++;
			try {

				Map.Entry entry = (Map.Entry)iterator.next();

				jobid = (Long)entry.getKey();

				ConcurrentHashMap<Long, SimpleTaskProcessor> simpleTaskProcessorMap = (ConcurrentHashMap<Long, SimpleTaskProcessor>)entry.getValue();

				if(CollectionUtils.isEmpty(simpleTaskProcessorMap)) {
					continue;
				}

				Iterator processorIterator = simpleTaskProcessorMap.entrySet().iterator();

				while(processorIterator.hasNext()) {
					processorCount++;
					Map.Entry processorEntity = (Map.Entry)processorIterator.next();
					SimpleTaskProcessor processor = (SimpleTaskProcessor)processorEntity.getValue();
					processor.kill();
				}

			} catch (Throwable e) {

				logger.error("[SimplePool]: stopService error, jobid:" + jobid, e);
			}

			logger.info("[SimplePool]: stopService, jobCount:"+jobCount +
					    ", processorCount:" + processorCount);
		}

		simpleJobTable.clear();

	}


	public boolean executeTask(ExecutableTask executableTask) {
		try {
			logger.info("executeTask start......");
			ConcurrentHashMap<Long, SimpleTaskProcessor> instanceTable = this.simpleJobTable.get(executableTask.getJob().getId());
			if(null == instanceTable) {
				instanceTable = new ConcurrentHashMap<Long, SimpleTaskProcessor>();

				ConcurrentHashMap<Long, SimpleTaskProcessor> existInstanceTable =
				this.simpleJobTable.put(executableTask.getJob().getId(), instanceTable);
				if (existInstanceTable!=null){
					instanceTable = existInstanceTable;
				}
			}
			SimpleTaskProcessor simpleTaskProcessor = new SimpleTaskProcessor(this.clientContext, executableTask);
			simpleTaskProcessor.setRpcContext(EagleEye.getRpcContext());
			simpleTaskProcessor.start();
			instanceTable.put(executableTask.getJobInstanceSnapshot().getId(), simpleTaskProcessor);
		} catch (Throwable e) {
			logger.error("[SimplePool]: executeTask error, instanceId:" + executableTask.getJobInstanceSnapshot().getId(), e);
			return false;
		}
		return true;
	}


	public int getInstanceAmount(Job job) {
		ConcurrentHashMap<Long, SimpleTaskProcessor> instanceTable = this.simpleJobTable.get(job.getId());
		if(null == instanceTable) {
			return 0;
		}
		return instanceTable.size();
	}


	public boolean stopTask(long jobId, long jobInstanceId) {
		ConcurrentHashMap<Long, SimpleTaskProcessor> instanceTable = this.simpleJobTable.get(jobId);
		if(null == instanceTable) {
			return true;
		}
		SimpleTaskProcessor simpleTaskProcessor = instanceTable.get(jobInstanceId);
		if(null == simpleTaskProcessor) {
			return true;
		}

		/** 删除任务 */
		try {
			Job job = new Job();
			job.setId(jobId);
			JobInstanceSnapshot jobInstanceSnapshot = new JobInstanceSnapshot();
			jobInstanceSnapshot.setId(jobInstanceId);
			this.removeTask(new ExecutableTask(job, jobInstanceSnapshot));
		} catch (Throwable e) {
			logger.error("[SimplePool]: stopTask removeTask error, jobId:" + jobId + ", jobInstanceId:" + jobInstanceId, e);
			return false;
		}
		return true;
	}


	@SuppressWarnings("deprecation")
	public boolean forceStopTask(long jobId, long jobInstanceId) {
		ConcurrentHashMap<Long, SimpleTaskProcessor> instanceTable = this.simpleJobTable.get(jobId);
		if(null == instanceTable) {
			return true;
		}
		SimpleTaskProcessor simpleTaskProcessor = instanceTable.get(jobInstanceId);
		if(null == simpleTaskProcessor) {
			return true;
		}

		try {
			simpleTaskProcessor.stop();
		} catch (Throwable e) {
			logger.error("[SimplePool]: forceStopTask error"
					+ ", jobId:" + jobId
					+ ", jobInstanceId:" + jobInstanceId, e);
		}

		/** 删除任务 */
		try {
			Job job = new Job();
			job.setId(jobId);
			JobInstanceSnapshot jobInstanceSnapshot = new JobInstanceSnapshot();
			jobInstanceSnapshot.setId(jobInstanceId);
			this.removeTask(new ExecutableTask(job, jobInstanceSnapshot));
		} catch (Throwable e) {
			logger.error("[SimplePool]: forceStopTask removeTask error"
					+ ", jobId:" + jobId
					+ ", jobInstanceId:" + jobInstanceId, e);
			return false;
		}
		return true;
	}


	public void removeTask(ExecutableTask executableTask) {
		try {
			ConcurrentHashMap<Long, SimpleTaskProcessor> instanceTable = this.simpleJobTable.get(executableTask.getJob().getId());
			if(null == instanceTable) {
				logger.warn("[SimplePool]: removeTask warn instanceTable is null, instanceId:" + executableTask.getJobInstanceSnapshot().getId());
				return ;
			}
			SimpleTaskProcessor simpleTaskProcessor = instanceTable.get(executableTask.getJobInstanceSnapshot().getId());
			if(null == simpleTaskProcessor) {
				logger.warn("[SimplePool]: removeTask warn simpleTaskProcessor is null, instanceId:" + executableTask.getJobInstanceSnapshot().getId());
				return ;
			}

			simpleTaskProcessor.kill();

			/** 删除Job实例 */
			instanceTable.remove(executableTask.getJobInstanceSnapshot().getId());

			if(instanceTable.size() <= 0) {
				this.simpleJobTable.remove(executableTask.getJob().getId());
			}
		} catch (Throwable e) {
			logger.error("[SimplePool]: removeTask error, instanceId:" + executableTask.getJobInstanceSnapshot().getId(), e);
		}
	}

	public Result<String> heartBeatCheckJobInstance(long jobId, long jobInstanceId) {
		Result<String> result = new Result<String>();
		ConcurrentHashMap<Long, SimpleTaskProcessor> instanceTable = this.simpleJobTable.get(jobId);
		if(null == instanceTable) {
			result.setResultCode(ResultCode.HEART_BEAT_CHECK_FAILURE);
			return result;
		}
		SimpleTaskProcessor simpleTaskProcessor = instanceTable.get(jobInstanceId);
		if(null == simpleTaskProcessor) {
			JobRunningStateManager.getManageHandler().deleteJobRunningState(jobInstanceId);
			result.setResultCode(ResultCode.HEART_BEAT_CHECK_FAILURE);
			return result;
		}
		result.setResultCode(ResultCode.HEART_BEAT_CHECK_SUCCESS);
		result.setData(JobRunningStateManager.getManageHandler().getJobRunningStateDescribe(jobInstanceId));
		return result;
	}

	public ConcurrentHashMap<Long, ConcurrentHashMap<Long, SimpleTaskProcessor>> getSimpleJobTable() {
		return simpleJobTable;
	}

}
