package com.alibaba.dts.client.executor.simple;

import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import com.alibaba.dts.client.executor.job.context.ClientContextImpl;
import com.alibaba.dts.client.executor.simple.processor.ScxSimpleTaskProcessor;
import com.alibaba.dts.client.executor.simple.processor.SimpleTaskProcessor;
import com.alibaba.dts.client.service.JobRunningStateManager;
import com.alibaba.dts.common.domain.ExecutableTask;
import com.alibaba.dts.common.domain.result.Result;
import com.alibaba.dts.common.domain.result.ResultCode;
import com.alibaba.dts.common.domain.store.Job;
import com.alibaba.dts.common.domain.store.JobInstanceSnapshot;
import com.alibaba.dts.common.logger.SchedulerXLoggerFactory;
import com.alibaba.dts.common.logger.innerlog.Logger;
import org.springframework.util.CollectionUtils;

/**
 * 简单job执行池
 * @author tianyao.myc
 *
 */
public class ScxSimplePool {

	private static final Logger logger = SchedulerXLoggerFactory.getLogger(ScxSimplePool.class);
	
	/** 简单Job映射表 */
	private ConcurrentHashMap<Long, ConcurrentHashMap<Long, ScxSimpleTaskProcessor>> scxSimpleJobTable =
			new ConcurrentHashMap<Long, ConcurrentHashMap<Long, ScxSimpleTaskProcessor>>();
	
	private final ClientContextImpl clientContext;
	
	public ScxSimplePool(final ClientContextImpl clientContext) {
		this.clientContext = clientContext;
	}

	public void stopService(){

		Iterator iterator = scxSimpleJobTable.entrySet().iterator();

		long jobCount = 0;
		long processorCount = 0;

		while(iterator.hasNext()) {

			long jobid=0;
			jobCount++;
			try {

				Map.Entry entry = (Map.Entry)iterator.next();

				jobid = (Long)entry.getKey();

				ConcurrentHashMap<Long, ScxSimpleTaskProcessor> scxSimpleTaskProcessorMap = (ConcurrentHashMap<Long, ScxSimpleTaskProcessor>)entry.getValue();

				if(CollectionUtils.isEmpty(scxSimpleTaskProcessorMap)) {
					continue;
				}

				Iterator processorIterator = scxSimpleTaskProcessorMap.entrySet().iterator();

				while(processorIterator.hasNext()) {
					processorCount++;
					Map.Entry processorEntity = (Map.Entry)processorIterator.next();
					ScxSimpleTaskProcessor processor = (ScxSimpleTaskProcessor)processorEntity.getValue();
					processor.setStop(true);
				}

			} catch (Throwable e) {

				logger.error("[ScxSimplePool]: stopService error, jobid:" + jobid, e);
			}

			logger.info("[ScxSimplePool]: stopService, jobCount:"+jobCount +
					", processorCount:" + processorCount);
		}

		scxSimpleJobTable.clear();
	}
	

	public boolean executeTask(ExecutableTask executableTask) {
		try {
			logger.info("executeTask start......");
			ConcurrentHashMap<Long, ScxSimpleTaskProcessor> instanceTable = this.scxSimpleJobTable.get(executableTask.getJob().getId());
			if(null == instanceTable) {
				instanceTable = new ConcurrentHashMap<Long, ScxSimpleTaskProcessor>();
				this.scxSimpleJobTable.put(executableTask.getJob().getId(), instanceTable);
			}
			ScxSimpleTaskProcessor scxSimpleTaskProcessor = new ScxSimpleTaskProcessor(this.clientContext, executableTask);
			scxSimpleTaskProcessor.start();
			instanceTable.put(executableTask.getJobInstanceSnapshot().getId(), scxSimpleTaskProcessor);
		} catch (Throwable e) {
			logger.error("[ScxSimplePool]: executeTask error, instanceId:" + executableTask.getJobInstanceSnapshot().getId(), e);
			return false;
		}
		return true;
	}
	

	public int getInstanceAmount(Job job) {
		ConcurrentHashMap<Long, ScxSimpleTaskProcessor> instanceTable = this.scxSimpleJobTable.get(job.getId());
		if(null == instanceTable) {
			return 0;
		}
		return instanceTable.size();
	}
	

	public boolean stopTask(long jobId, long jobInstanceId) {
		ConcurrentHashMap<Long, ScxSimpleTaskProcessor> instanceTable = this.scxSimpleJobTable.get(jobId);
		if(null == instanceTable) {
			return true;
		}
		ScxSimpleTaskProcessor scxSimpleTaskProcessor = instanceTable.get(jobInstanceId);
		if(null == scxSimpleTaskProcessor) {
			return true;
		}
		
		/** 删除任务 */
		try {
			Job job = new Job();
			job.setId(jobId);
			JobInstanceSnapshot jobInstanceSnapshot = new JobInstanceSnapshot();
			jobInstanceSnapshot.setId(jobInstanceId);
			this.removeTask(new ExecutableTask(job, jobInstanceSnapshot));
		} catch (Throwable e) {
			logger.error("[ScxSimplePool]: stopTask removeTask error, jobId:" + jobId + ", jobInstanceId:" + jobInstanceId, e);
			return false;
		}
		return true;
	}
	

	@SuppressWarnings("deprecation")
	public boolean forceStopTask(long jobId, long jobInstanceId) {
		ConcurrentHashMap<Long, ScxSimpleTaskProcessor> instanceTable = this.scxSimpleJobTable.get(jobId);
		if(null == instanceTable) {
			return true;
		}
		ScxSimpleTaskProcessor scxSimpleTaskProcessor = instanceTable.get(jobInstanceId);
		if(null == scxSimpleTaskProcessor) {
			return true;
		}
		
		try {
			scxSimpleTaskProcessor.stop();
		} catch (Throwable e) {
			logger.error("[ScxSimplePool]: forceStopTask error"
					+ ", jobId:" + jobId 
					+ ", jobInstanceId:" + jobInstanceId, e);
		}
		
		/** 删除任务 */
		try {
			Job job = new Job();
			job.setId(jobId);
			JobInstanceSnapshot jobInstanceSnapshot = new JobInstanceSnapshot();
			jobInstanceSnapshot.setId(jobInstanceId);
			this.removeTask(new ExecutableTask(job, jobInstanceSnapshot));
		} catch (Throwable e) {
			logger.error("[ScxSimplePool]: forceStopTask removeTask error"
					+ ", jobId:" + jobId 
					+ ", jobInstanceId:" + jobInstanceId, e);
			return false;
		}
		return true;
	}
	

	public void removeTask(ExecutableTask executableTask) {
		try {
			ConcurrentHashMap<Long, ScxSimpleTaskProcessor> instanceTable = this.scxSimpleJobTable.get(executableTask.getJob().getId());
			if(null == instanceTable) {
				logger.warn("[ScxSimplePool]: removeTask warn instanceTable is null, instanceId:" + executableTask.getJobInstanceSnapshot().getId());
				return ;
			}
			ScxSimpleTaskProcessor scxSimpleTaskProcessor = instanceTable.get(executableTask.getJobInstanceSnapshot().getId());
			if(null == scxSimpleTaskProcessor) {
				logger.warn("[ScxSimplePool]: removeTask warn simpleTaskProcessor is null, instanceId:" + executableTask.getJobInstanceSnapshot().getId());
				return ;
			}
			
			/** 删除Job实例 */
			instanceTable.remove(executableTask.getJobInstanceSnapshot().getId());
			
			if(instanceTable.size() <= 0) {
				this.scxSimpleJobTable.remove(executableTask.getJob().getId());
			}
		} catch (Throwable e) {
			logger.error("[ScxSimplePool]: removeTask error, instanceId:" + executableTask.getJobInstanceSnapshot().getId(), e);
		}
	}


	public Result<String> heartBeatCheckJobInstance(long jobId, long jobInstanceId) {
		Result<String> result = new Result<String>();
		ConcurrentHashMap<Long, ScxSimpleTaskProcessor> instanceTable = this.scxSimpleJobTable.get(jobId);
		if(null == instanceTable) {
			result.setResultCode(ResultCode.HEART_BEAT_CHECK_FAILURE);
			return result;
		}
		ScxSimpleTaskProcessor scxSimpleTaskProcessor = instanceTable.get(jobInstanceId);
		if(null == scxSimpleTaskProcessor) {
			JobRunningStateManager.getManageHandler().deleteJobRunningState(jobInstanceId);
			result.setResultCode(ResultCode.HEART_BEAT_CHECK_FAILURE);
			return result;
		}
		result.setResultCode(ResultCode.HEART_BEAT_CHECK_SUCCESS);
		result.setData(JobRunningStateManager.getManageHandler().getJobRunningStateDescribe(jobInstanceId));
		return result;
	}
	
	public ConcurrentHashMap<Long, ConcurrentHashMap<Long, ScxSimpleTaskProcessor>> getSimpleJobTable() {
		return scxSimpleJobTable;
	}
	
}
