package com.alibaba.dts.client.executor.parallel.unit;

import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import org.springframework.util.CollectionUtils;

import com.alibaba.dts.client.executor.job.context.ClientContextImpl;
import com.alibaba.dts.client.executor.parallel.ParallelPool;
import com.alibaba.dts.client.executor.parallel.processor.ParallelTaskProcessor;
import com.alibaba.dts.client.executor.parallel.processor.PullProcessor;
import com.alibaba.dts.common.constants.Constants;
import com.alibaba.dts.common.domain.ExecutableTask;
import com.alibaba.dts.common.domain.store.TaskSnapshot;
import com.alibaba.dts.common.exception.InitException;
import com.alibaba.dts.common.logger.SchedulerXLoggerFactory;
import com.alibaba.dts.common.logger.innerlog.Logger;

/**
 * 执行单元
 * @author tianyao.myc
 *
 */
public class ExecutorUnit implements Constants {

	private static final Logger logger = SchedulerXLoggerFactory.getLogger(ExecutorUnit.class);

	/** 执行单元实例标识 */
	private ExecutableTask executableTask;

	/** 拉任务快照线程 */
	private PullProcessor pullProcessor = null;

	/** 任务队列 */
	private BlockingQueue<TaskSnapshot> queue = null;


	/** 任务处理器线程组 */
	private ParallelTaskProcessor[] parallelTaskProcessors = null;

	/** 线程计数器 */
	private final AtomicInteger threadCounter = new AtomicInteger();

	private final ParallelPool parallelPool;

	private final ClientContextImpl clientContext;

	public ExecutorUnit(final ClientContextImpl clientContext, ParallelPool parallelPool, ExecutableTask executableTask) {
		this.clientContext = clientContext;
		this.parallelPool = parallelPool;
		this.executableTask = executableTask;

		int pageSize = this.clientContext.getClientConfig().getPageSize();
		Map<String, Integer> pageSizeMap = this.clientContext.getClientConfig().getPageSizeMap();
		if(! CollectionUtils.isEmpty(pageSizeMap) && pageSizeMap.get(executableTask.getJob().getJobProcessor()) != null) {
			pageSize = this.clientContext.getClientConfig().checkPageSize(pageSizeMap.get(executableTask.getJob().getJobProcessor()).intValue());
		}
		this.executableTask.setLength(pageSize);
	}

	/**
	 * 刷新执行单元信息
	 *  executableTask
	 */
	public void refresh(ExecutableTask executableTask) {
		this.executableTask = executableTask;

		int pageSize = this.clientContext.getClientConfig().getPageSize();
		Map<String, Integer> pageSizeMap = this.clientContext.getClientConfig().getPageSizeMap();
		if(! CollectionUtils.isEmpty(pageSizeMap) && pageSizeMap.get(executableTask.getJob().getJobProcessor()) != null) {
			pageSize = this.clientContext.getClientConfig().checkPageSize(pageSizeMap.get(executableTask.getJob().getJobProcessor()).intValue());
		}
		this.executableTask.setLength(pageSize);

		for(int i = 0 ; i < this.parallelTaskProcessors.length ; i ++) {
			this.parallelTaskProcessors[i].refresh(this, i);
		}

		this.pullProcessor.refresh(this);
	}

	/**
	 * 初始化
	 *  com.alibaba.dts.common.exception.InitException
	 */
	public void init() throws InitException {

		/** 初始化拉任务快照线程 */
		this.pullProcessor = new PullProcessor(this.clientContext, this);

		/** 初始化任务队列 */
		this.queue = new LinkedBlockingQueue<TaskSnapshot>(this.clientContext.getClientConfig().getQueueSize());

		this.pullProcessor.start();

		int consumerThreads = this.clientContext.getClientConfig().getConsumerThreads();
		Map<String, Integer> consumerThreadsMap = this.clientContext.getClientConfig().getConsumerThreadsMap();
		if(! CollectionUtils.isEmpty(consumerThreadsMap) && consumerThreadsMap.get(executableTask.getJob().getJobProcessor()) != null) {
			consumerThreads = this.clientContext.getClientConfig().checkConsumerThreads(consumerThreadsMap.get(executableTask.getJob().getJobProcessor()).intValue());
		}

		if(executableTask.getRunThreads() > 0) {
			consumerThreads = executableTask.getRunThreads();
		}

		/** 初始化任务处理线程组 */
		this.parallelTaskProcessors = new ParallelTaskProcessor[consumerThreads];
		for(int i = 0 ; i < consumerThreads ; i ++) {
			this.parallelTaskProcessors[i] = new ParallelTaskProcessor(this.clientContext, this, i, this.threadCounter);
			this.parallelTaskProcessors[i].start();
		}
	}

	/**
	 * 清空队列
	 */
	public void clear() {
		try {
			this.queue.clear();
		} catch (Throwable e) {
			logger.error("[ExecutorUnit]: clear error"
					+ ", instanceId:" + this.executableTask.getJobInstanceSnapshot().getId(), e);
		}
	}

	/**
	 * 停止任务
	 */
	public void stopTask() {

		try{
			/** 强制停止拉取线程 */
			if(pullProcessor != null) {
				pullProcessor.setStop(true);
			}

			/** 强制停止执行任务线程 */
			for(int i = 0 ; i < this.parallelTaskProcessors.length ; i ++) {
				if(this.parallelTaskProcessors[i] != null) {
					this.parallelTaskProcessors[i].setStop(true);
				}
			}

			clear();

		}catch (Throwable e) {
			logger.error("[ExecutorUnit]: stopTask error"
					+ ", instanceId:" + this.executableTask.getJobInstanceSnapshot().getId(), e);
		}
	}

	/**
	 * 强制停止
	 */
	@SuppressWarnings("deprecation")
	public void forceStopTask() {

		try {
			pullProcessor.stop();
		} catch (Throwable e) {
			logger.error("[ExecutorUnit]: forceStopTask pullProcessor error"
					+ ", instanceId:" + this.executableTask.getJobInstanceSnapshot().getId(), e);
		}

		//清空队列
		clear();

		/** 强制停止执行任务线程 */
		for(int i = 0 ; i < this.parallelTaskProcessors.length ; i ++) {
			try {
				this.parallelTaskProcessors[i].stop();
			} catch (Throwable e) {
				logger.error("[ExecutorUnit]: forceStopTask parallelTaskProcessors error"
						+ ", instanceId:" + this.executableTask.getJobInstanceSnapshot().getId(), e);
			}
		}

	}

	/**
	 * 执行器是否停止
	 *
	 */
	public boolean isExecutorStop() {

		if (queue==null){
			return true;
		}
		return  queue.isEmpty() && (threadCounter.get() == 0);
	}

	/**
	 * 将任务放入队列
	 *  taskSnapshot
	 *
	 */
	public boolean offer(TaskSnapshot taskSnapshot) {

		boolean result = false;
		try {
			result = queue.offer(taskSnapshot, DEFAULT_INVOKE_TIMEOUT, TimeUnit.MILLISECONDS);
		} catch (Throwable e) {
			logger.error("[ExecutorUnit]: offer error"
					+ ", jobInstanceId:" + taskSnapshot.getJobInstanceId()
					+ ", id:" + taskSnapshot.getId(), e);
		}

		return result;
	}

	public boolean isExistsProcessors(){
		Boolean result=false;
		if (this.parallelTaskProcessors == null){
			return false;
		}else{
			for(int i = 0 ; i < this.parallelTaskProcessors.length ; i ++) {
				if (!this.parallelTaskProcessors[i].isStop())  {
					return true;
				}
			}
		}
		return result;
	}

	public ExecutableTask getExecutableTask() {
		return executableTask;
	}

	public BlockingQueue<TaskSnapshot> getQueue() {
		return queue;
	}


	public ParallelTaskProcessor[] getParallelTaskProcessors() {
		return parallelTaskProcessors;
	}

	public AtomicInteger getThreadCounter() {
		return threadCounter;
	}

	public ParallelPool getParallelPool() {
		return parallelPool;
	}

	@Override
	public String toString() {
		return "ExecutorUnit [executableTask=" + executableTask + "]";
	}

}
