package com.alibaba.dts.client.executor.parallel.processor;

import java.util.Collections;
import java.util.Date;
import java.util.List;

import org.springframework.util.CollectionUtils;

import com.alibaba.dts.client.executor.job.context.ClientContextImpl;
import com.alibaba.dts.client.executor.job.context.JobContextImpl;
import com.alibaba.dts.common.constants.Constants;
import com.alibaba.dts.common.context.InvocationContext;
import com.alibaba.dts.common.domain.ExecutableTask;
import com.alibaba.dts.common.domain.remoting.RemoteMachine;
import com.alibaba.dts.common.domain.result.Result;
import com.alibaba.dts.common.domain.result.ResultCode;
import com.alibaba.dts.common.domain.store.Job;
import com.alibaba.dts.common.domain.store.JobInstanceSnapshot;
import com.alibaba.dts.common.domain.store.TaskSnapshot;
import com.alibaba.dts.common.logger.SchedulerXLoggerFactory;
import com.alibaba.dts.common.logger.innerlog.Logger;
import com.alibaba.dts.common.service.ServerService;
import com.alibaba.dts.common.util.BytesUtil;
import com.alibaba.dts.common.util.BytesUtil4Client;
import com.alibaba.dts.common.util.RandomUtil;
import com.alibaba.dts.common.util.StringUtil;

/**
 * Created by yif on 16/8/30.
 */
public class ParallelJobContextImpl extends JobContextImpl implements Constants,ParallelJobContext {
    private static final Logger logger = SchedulerXLoggerFactory.getLogger(ParallelJobContextImpl.class);

    /**
     * 当前要处理的任务快照
     */
    private TaskSnapshot taskSnapshot;

    /**
     * 当前要处理的任务
     */
    private Object task;

    /**
     * 任务名称
     */
    private String taskName;

    //可用的机器数量
    private int availableMachineAmount;

    //当前机器编号
    private int currentMachineNumber;

    private ServerService serverService;

    private final ClientContextImpl clientContext;

    public ParallelJobContextImpl(final ClientContextImpl clientContext, Job job, JobInstanceSnapshot jobInstanceSnapshot, int retryCount) {
        super(job, jobInstanceSnapshot, retryCount);

        this.clientContext = clientContext;
        this.serverService = this.clientContext.getClientRemoting().proxyInterface(ServerService.class);
    }

    /**
     * 初始化重试次数
     *
     *  retryCount
     */
    public void initRetryCount(int retryCount) {
        super.setRetryCount(retryCount);
    }

    /**
     * 设置任务
     *
     *  taskSnapshot
     */
    public void setTask(TaskSnapshot taskSnapshot) {
        this.taskSnapshot = taskSnapshot;
        this.taskName = taskSnapshot.getTaskName();
        if (DEFAULT_ROOT_LEVEL_TASK_NAME.equals(taskSnapshot.getTaskName())) {
            if (BytesUtil.isEmpty(taskSnapshot.getBody())) {
                logger.error("[ParallelJobContext]: BytesUtil setTask bytesToObject error, body is empty"
                        + ", instanceId:" + taskSnapshot.getJobInstanceId()
                        + ", id:" + taskSnapshot.getId());
                return;
            }
            try {
                this.task = BytesUtil.bytesToObject(taskSnapshot.getBody());
            } catch (Throwable e) {
                logger.error("[ParallelJobContext]: BytesUtil setTask bytesToObject error"
                        + ", instanceId:" + taskSnapshot.getJobInstanceId()
                        + ", id:" + taskSnapshot.getId(), e);
            }
        } else {
            if (BytesUtil4Client.isEmpty(taskSnapshot.getBody())) {
                logger.error("[ParallelJobContext]: BytesUtil4Client setTask bytesToObject error, body is empty"
                        + ", instanceId:" + taskSnapshot.getJobInstanceId()
                        + ", id:" + taskSnapshot.getId());
                return;
            }
            try {
                this.task = BytesUtil4Client.bytesToObject(taskSnapshot.getBody());
            } catch (Throwable e) {
                logger.error("[ParallelJobContext]: BytesUtil4Client setTask bytesToObject error"
                        + ", instanceId:" + taskSnapshot.getJobInstanceId()
                        + ", id:" + taskSnapshot.getId(), e);
            }
        }
    }

    /**
     * 分发任务列表
     *
     *  taskList 任务列表
     *  taskName 任务名称
     *
     */
    public Result<Boolean> dispatchTaskList(List<? extends Object> taskList, String taskName) {
        Result<Boolean> result = new Result<Boolean>(false);

        if (StringUtil.isBlank(taskName)) {
            logger.error("[ParallelJobContext]: dispatchTaskList taskName is isEmpty error"
                    + ", jobId:" + job.getId());
            result.setResultCode(ResultCode.DISPATCH_TASK_LIST_NAME_IS_NULL);
            return result;
        }

        if (CollectionUtils.isEmpty(taskList)) {
            logger.warn("[ParallelJobContext]: dispatchTaskList taskList is empty"
                    + ", taskName:" + taskName
                    + ", jobId:" + job.getId());
            result.setResultCode(ResultCode.DISPATCH_TASK_LIST_IS_EMPTY);
            return result;
        }

        if (taskList.size() > 3000) {
            throw new RuntimeException("taskList size too large, max:" + 3000 + ", but you set " + taskList.size());
        }

        List<String> serverList = this.clientContext.getClientRemoting().getServerList();
        if (CollectionUtils.isEmpty(serverList)) {
            logger.error("[ParallelJobContext]: dispatchTaskList serverList is isEmpty error"
                    + ", taskName:" + taskName
                    + ", jobId:" + job.getId());
            result.setResultCode(ResultCode.DISPATCH_TASK_LIST_SERVER_DOWN);
            return result;
        }

        ExecutableTask executableTask = new ExecutableTask(job, jobInstanceSnapshot);
        executableTask.setTaskSnapshot(this.taskSnapshot);
        for (Object task : taskList) {
            fillTaskSnapshot(executableTask, task, taskName);
        }

        /** 随机列表顺序 */
        Collections.shuffle(serverList);

        for (String server : serverList) {

            Result<Boolean> sendResult = null;
            try {
                InvocationContext.setRemoteMachine(new RemoteMachine(server, 12 * DEFAULT_INVOKE_TIMEOUT));
                sendResult = serverService.send(executableTask);
            } catch (Throwable e) {
                logger.error("[ParallelJobContext]: dispatchTaskList sendOld error"
                        + ", taskName:" + taskName
                        + ", jobId:" + job.getId()
                        + ", server:" + server, e);
            }

            if (null == sendResult || !sendResult.getData().booleanValue()) {
                logger.error("[ParallelJobContext]: dispatchTaskList sendOld failed"
                        + ", taskName:" + taskName
                        + ", jobId:" + job.getId()
                        + ", server:" + server);
                result.setData(null == sendResult ? false : sendResult.getData());
                result.setResultCode(null == sendResult ? ResultCode.DISPATCH_TASK_LIST_SERVER_DO_NOT_RESPONSE : sendResult.getResultCode());
            } else {
                result.setData(sendResult.getData());
                result.setResultCode(sendResult.getResultCode());
                return result;
            }

        }

        return result;
    }

    /**
     * 装填任务参数
     *
     *  executableTask
     *  task
     *  taskName
     */
    public void fillTaskSnapshot(ExecutableTask executableTask, Object task, String taskName) {
        byte[] body = null;
        try {
            body = BytesUtil4Client.objectToBytes(task);
        } catch (Throwable e) {
            logger.error("[ParallelJobContext]: fillTaskSnapshot objectToBytes error"
                    + ", taskName:" + taskName
                    + ", jobId:" + job.getId()
                    + ", task:" + task, e);
        }
        if (BytesUtil4Client.isEmpty(body)) {
            logger.error("[ParallelJobContext]: fillTaskSnapshot objectToBytes body is empty"
                    + ", taskName:" + taskName
                    + ", jobId:" + job.getId()
                    + ", task:" + task);
            return;
        }

        if (body.length > this.clientContext.getClientConfig().getMaxBodySize()) {
            throw new RuntimeException("[ParallelJobContext]: single task is too large, more than 64KB");
        }

        TaskSnapshot taskSnapshot = new TaskSnapshot();
        taskSnapshot.setGmtCreate(new Date());
        taskSnapshot.setGmtModified(new Date());
        taskSnapshot.setJobInstanceId(executableTask.getJobInstanceSnapshot().getId());
        taskSnapshot.setJobProcessor(executableTask.getJob().getJobProcessor());
        taskSnapshot.setBody(body);
        taskSnapshot.setStatus(TASK_STATUS_INIT);
        taskSnapshot.setTaskName(taskName);
        taskSnapshot.setRetryCount(0);

        executableTask.addTaskSnapshot(taskSnapshot);
    }

    public Object getTask() {
        return task;
    }

    public String getTaskName() {
        return taskName;
    }

    /**
     * 获取自定义全局变量
     *
     *
     */
    public String getGlobalArguments() {

        List<String> serverList = this.clientContext.getClientRemoting().getServerList();
        if (CollectionUtils.isEmpty(serverList)) {
            logger.error("[ParallelJobContext]: getGlobalArguments serverList is isEmpty error"
                    + ", instanceId:" + jobInstanceSnapshot.getId());
            return null;
        }

        InvocationContext.setRemoteMachine(new RemoteMachine(RandomUtil.getRandomObj(serverList)));
        Result<String> body = serverService.getGlobalArguments(jobInstanceSnapshot);
        if (null == body) {
            logger.error("[ParallelJobContext]: getGlobalArguments body is null"
                    + ", instanceId:" + jobInstanceSnapshot.getId());
            return null;
        }

        return body.getData();
    }

    /**
     * 设置自定义全局变量
     *
     *  globalArguments
     *
     */
    public Result<Boolean> setGlobalArguments(String globalArguments) {
        Result<Boolean> result = new Result<Boolean>(false);
        if (StringUtil.isBlank(globalArguments)) {
            result.setResultCode(ResultCode.SET_GLOBAL_ARGUMENTS_NULL);
            return result;
        }

        List<String> serverList = this.clientContext.getClientRemoting().getServerList();
        if (CollectionUtils.isEmpty(serverList)) {
            logger.error("[ParallelJobContext]: setGlobalArguments serverList is isEmpty error"
                    + ", instanceId:" + jobInstanceSnapshot.getId());
            result.setResultCode(ResultCode.SET_GLOBAL_SERVER_DOWN);
            return result;
        }

        InvocationContext.setRemoteMachine(new RemoteMachine(RandomUtil.getRandomObj(serverList)));
        Result<Boolean> setResult = serverService.setGlobalArguments(jobInstanceSnapshot, globalArguments);
        if (null == setResult) {
            result.setResultCode(ResultCode.SET_GLOBAL_FAILURE);
            return result;
        }

        result.setData(setResult.getData());
        result.setResultCode(setResult.getResultCode());
        return result;
    }

    public int getAvailableMachineAmount() {
        return availableMachineAmount;
    }

    public void setAvailableMachineAmount(int availableMachineAmount) {
        this.availableMachineAmount = availableMachineAmount;
    }

    public int getCurrentMachineNumber() {
        return currentMachineNumber;
    }

    public void setCurrentMachineNumber(int currentMachineNumber) {
        this.currentMachineNumber = currentMachineNumber;
    }
}
