package com.alibaba.dts.client.executor.parallel;

import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.springframework.util.CollectionUtils;

import com.alibaba.dts.client.executor.job.context.ClientContextImpl;
import com.alibaba.dts.client.executor.parallel.unit.ExecutorUnit;
import com.alibaba.dts.common.constants.Constants;
import com.alibaba.dts.common.domain.ExecutableTask;
import com.alibaba.dts.common.domain.result.Result;
import com.alibaba.dts.common.domain.result.ResultCode;
import com.alibaba.dts.common.domain.store.TaskSnapshot;
import com.alibaba.dts.common.logger.SchedulerXLoggerFactory;
import com.alibaba.dts.common.logger.innerlog.Logger;

/**
 * 并行计算job执行池
 * @author tianyao.myc
 *
 */
public class ParallelPool implements Constants {

	private static final Logger logger = SchedulerXLoggerFactory.getLogger(ParallelPool.class);

	/** 任务执行单元映射表 */
	private ConcurrentHashMap<Long, ConcurrentHashMap<Long, ExecutorUnit>> executorUnitTable =
			new ConcurrentHashMap<Long, ConcurrentHashMap<Long, ExecutorUnit>>();

	private final ClientContextImpl clientContext;

	public ParallelPool(final ClientContextImpl clientContext) {
		this.clientContext = clientContext;
	}

	public void stopService(){

		Iterator iterator = executorUnitTable.entrySet().iterator();

		long jobCount = 0;
		long processorCount = 0;

		while(iterator.hasNext()) {

			long jobid=0;
			jobCount++;
			try {

				Map.Entry entry = (Map.Entry)iterator.next();

				jobid = (Long)entry.getKey();

				ConcurrentHashMap<Long, ExecutorUnit> executorUnitMap = (ConcurrentHashMap<Long, ExecutorUnit>)entry.getValue();

				if(CollectionUtils.isEmpty(executorUnitMap)) {
					continue;
				}

				Iterator executorUnitIterator = executorUnitMap.entrySet().iterator();

				while(executorUnitIterator.hasNext()) {
					processorCount++;
					Map.Entry processorEntity = (Map.Entry)executorUnitIterator.next();
					ExecutorUnit processor = (ExecutorUnit)processorEntity.getValue();
					processor.stopTask();
				}

			} catch (Throwable e) {

				logger.error("[SimplePool]: stopService error, jobid:" + jobid, e);
			}

			logger.info("[SimplePool]: stopService, jobCount:"+jobCount +
					", processorCount:" + processorCount);
		}

		executorUnitTable.clear();

	}


	public boolean executeTask(ExecutableTask executableTask) {
		ConcurrentHashMap<Long, ExecutorUnit> executorUnitMap = this.executorUnitTable.get(executableTask.getJob().getId());
		if(null == executorUnitMap) {
			executorUnitMap = new ConcurrentHashMap<Long, ExecutorUnit>();
			ConcurrentHashMap<Long, ExecutorUnit> existExecutorUnitMap =
			this.executorUnitTable.putIfAbsent(executableTask.getJob().getId(), executorUnitMap);
			if (existExecutorUnitMap!=null){
				executorUnitMap = existExecutorUnitMap;
			}
		}

		ExecutorUnit executorUnit = executorUnitMap.get(executableTask.getJobInstanceSnapshot().getId());
		if(null == executorUnit) {
			executorUnit = new ExecutorUnit(this.clientContext, this, executableTask);

			ExecutorUnit existExecutorUnit =
			executorUnitMap.putIfAbsent(executableTask.getJobInstanceSnapshot().getId(), executorUnit);
			if (existExecutorUnit!=null){
				executorUnit = null;
			}else{
				try {
					executorUnit.init();
					logger.warn("[ParallelPool]: executeTask init, executableTask:" + executableTask);
				} catch (Throwable e) {
					logger.error("[ParallelPool]: executeTask init error"
							+ ", instanceId:" + executableTask.getJobInstanceSnapshot().getId(), e);
					return false;
				}
			}
		}

		return true;
	}


	public boolean stopTask(long jobId, long jobInstanceId) {
		ConcurrentHashMap<Long, ExecutorUnit> executorUnitMap = this.executorUnitTable.get(jobId);
		if(null == executorUnitMap || executorUnitMap.isEmpty()) {
			return true;
		}
		ExecutorUnit executorUnit = executorUnitMap.get(jobInstanceId);
		if(null == executorUnit) {
			return true;
		}

		/** 停止任务执行单元 */
		executorUnit.stopTask();

		if (!executorUnit.isExistsProcessors()){

			/** 删除任务执行单元 */
			try {
				executorUnitMap.remove(jobInstanceId);
				logger.info("removed jobInstanceId=" + jobInstanceId + " from executorUnitMap");
			} catch (Throwable e) {
				logger.error("[ParallelPool]: stopTask remove error"
						+ ", jobId:" + jobId
						+ ", jobInstanceId:" + jobInstanceId, e);
				return false;
			}

		}

		return true;
	}

	public boolean forceStopTask(long jobId, long jobInstanceId) {
		ConcurrentHashMap<Long, ExecutorUnit> executorUnitMap = this.executorUnitTable.get(jobId);
		if(null == executorUnitMap || executorUnitMap.isEmpty()) {
			return true;
		}
		ExecutorUnit executorUnit = executorUnitMap.get(jobInstanceId);
		if(null == executorUnit) {
			return true;
		}

		/** 强制停止任务执行单元 */
		executorUnit.forceStopTask();

		/** 删除任务执行单元 */
		try {
			executorUnitMap.remove(jobInstanceId);
		} catch (Throwable e) {
			logger.error("[ParallelPool]: forceStopTask remove error"
					+ ", jobId:" + jobId
					+ ", jobInstanceId:" + jobInstanceId, e);
			return false;
		}
		return true;
	}


	public Result<String> heartBeatCheckJobInstance(long jobId, long jobInstanceId) {
		Result<String> result = new Result<String>();
		ConcurrentHashMap<Long, ExecutorUnit> executorUnitMap = this.executorUnitTable.get(jobId);
		if(null == executorUnitMap || executorUnitMap.isEmpty()) {
			result.setResultCode(ResultCode.HEART_BEAT_CHECK_EXIT);
			return result;
		}
		ExecutorUnit executorUnit = executorUnitMap.get(jobInstanceId);
		if(null == executorUnit) {
			result.setResultCode(ResultCode.HEART_BEAT_CHECK_EXIT);
			return result;
		}
		if(executorUnit.isExecutorStop()) {
			result.setResultCode(ResultCode.HEART_BEAT_CHECK_FAILURE);
			return result;
		}
		result.setResultCode(ResultCode.HEART_BEAT_CHECK_SUCCESS);
		return result;
	}


	public Result<Boolean> push(long jobId, long jobInstanceId, TaskSnapshot taskSnapshot) {

		ConcurrentHashMap<Long, ExecutorUnit> executorUnitMap = this.executorUnitTable.get(jobId);
		if(null == executorUnitMap || executorUnitMap.isEmpty()) {
			return new Result<Boolean>(false, ResultCode.PUSH_UNIT_MAP_IS_EMPTY_ERROR);
		}

		ExecutorUnit executorUnit = executorUnitMap.get(jobInstanceId);
		if(null == executorUnit) {
			return new Result<Boolean>(false, ResultCode.PUSH_UNIT_IS_NULL_ERROR);
		}

		//将任务放入队列
		boolean result = executorUnit.offer(taskSnapshot);

		return new Result<Boolean>(result, result ? ResultCode.SUCCESS : ResultCode.FAILURE);
	}

}
