package com.alibaba.dts.client.executor.longtime.unit;

import java.util.Date;
import java.util.Map;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;

import org.springframework.util.CollectionUtils;

import com.alibaba.dts.client.executor.job.context.ClientContextImpl;
import com.alibaba.dts.client.executor.longtime.LongTimePool;
import com.alibaba.dts.client.executor.longtime.processor.LongTimeTaskProcessor;
import com.alibaba.dts.client.executor.longtime.processor.PullProcessor;
import com.alibaba.dts.client.executor.longtime.processor.ReFillingProcessor;
import com.alibaba.dts.common.constants.Constants;
import com.alibaba.dts.common.domain.ExecutableTask;
import com.alibaba.dts.common.domain.result.Result;
import com.alibaba.dts.common.domain.store.TaskSnapshot;
import com.alibaba.dts.common.exception.InitException;
import com.alibaba.dts.common.logger.SchedulerXLoggerFactory;
import com.alibaba.dts.common.logger.innerlog.Logger;

public class ExecutorUnit implements Constants {

	private static final Logger logger = SchedulerXLoggerFactory.getLogger(ExecutorUnit.class);

	private volatile boolean releaseTaskFlag = false;

	private volatile boolean pullTaskFlag = false;

	private final ClientContextImpl clientContext;
	
	/** 执行单元实例标识 */
	private ExecutableTask executableTask;
	
	/** 拉任务快照线程 */
	private PullProcessor pullProcessor = null;

	/** 重新装填任务快照线程 */
	private ReFillingProcessor reFillingProcessor = null;

	/** 任务队列 */
	private BlockingQueue<TaskSnapshot> queue = null;


	/** 完成任务队列 */
	private BlockingQueue<TaskSnapshot> completedqueue = null;
	
	/** 任务处理器线程组 */
	private LongTimeTaskProcessor[] longTimeTaskProcessors = null;

	private ConcurrentHashMap<Long,TaskRunStatistic> taskRunStatisticMap = new ConcurrentHashMap<Long,TaskRunStatistic>();

	private BlockingQueue<TaskSnapshot> releaseQueue = new LinkedBlockingQueue<TaskSnapshot>();

	/** 线程计数器 */
	private final AtomicInteger threadCounter = new AtomicInteger();

	private final LongTimePool longTimePool;

	private ScheduledExecutorService executorService = Executors
			.newScheduledThreadPool(1, new ThreadFactory() {
				public Thread newThread(Runnable runnable) {
					return new Thread(runnable, "DTS-LongTimeTaskStates-report-thread"+"-"
					+ executableTask.getJob().getId()+"-"
					+ executableTask.getJob().getJobProcessor()+"-"
					+ executableTask.getJobInstanceSnapshot().getId()
					);
				}
			});

	public ConcurrentHashMap<Long, TaskRunStatistic> getTaskRunStatisticMap() {
		return taskRunStatisticMap;
	}

	public ExecutorUnit(final ClientContextImpl clientContext, LongTimePool longTimePool, ExecutableTask executableTask) {
		this.clientContext = clientContext;
		this.longTimePool = longTimePool;
		this.executableTask = executableTask;

		int pageSize = clientContext.getClientConfig().getPageSize();
		Map<String, Integer> pageSizeMap = clientContext.getClientConfig().getPageSizeMap();
		if(! CollectionUtils.isEmpty(pageSizeMap) && pageSizeMap.get(executableTask.getJob().getJobProcessor()) != null) {
			pageSize = clientContext.getClientConfig().checkPageSize(pageSizeMap.get(executableTask.getJob().getJobProcessor()).intValue());
		}
		this.executableTask.setLength(1);

		releaseQueue = new LinkedBlockingQueue<TaskSnapshot>(MAX_LONGTIMETASKLIST_SIZE);
	}

	public boolean isPullTaskFlag() {
		return pullTaskFlag;
	}

	public void setPullTaskFlag(boolean pullTaskLock) {
		this.pullTaskFlag=pullTaskLock;
	}

	public boolean isReleaseTaskFlag() {
		return releaseTaskFlag;
	}

	public void setReleaseTaskFlag(boolean releaseTaskLock) {
		this.releaseTaskFlag = releaseTaskLock;
	}

	public boolean isExistsInTaskRunStatisticMap(Long taskid){
		return taskRunStatisticMap.containsKey(taskid);
	}

	public void addTaskRunStatisticMap(Long taskid){
		taskRunStatisticMap.put(taskid,new TaskRunStatistic(taskid));
	}

	public void addTaskRunStatisticMap(Long taskid,int lastProcessResult){
		taskRunStatisticMap.put(taskid,new TaskRunStatistic(taskid,lastProcessResult));
	}

	public void updateTaskRunStatisticMap(Long taskid,Long runtime){

		try{
			TaskRunStatistic taskRunStatistic = taskRunStatisticMap.get(taskid);

			taskRunStatistic.setRuntimes(taskRunStatistic.getRuntimes()+1);
			taskRunStatistic.setLastTimeConsuming(runtime);
			taskRunStatistic.setLastRunTime(new Date());
			taskRunStatisticMap.put(taskid,taskRunStatistic);

		}catch (Throwable e){
			logger.error("[LExecutorUnit]: updateTaskRunStatisticMap error"
					+ ", taskid:" + taskid
					+ ", runtime:" + runtime
					, e);
		}

	}

	public void updateTaskRunStatisticMap(Long taskid,Long runtime,int lastProcessResult){

		try{
			TaskRunStatistic taskRunStatistic = taskRunStatisticMap.get(taskid);

			taskRunStatistic.setRuntimes(taskRunStatistic.getRuntimes()+1);
			taskRunStatistic.setLastTimeConsuming(runtime);
			taskRunStatistic.setLastRunTime(new Date());
			taskRunStatistic.setProcessResult(lastProcessResult);

			taskRunStatisticMap.put(taskid,taskRunStatistic);
		}catch (Throwable e){
			logger.error("[LExecutorUnit]: updateTaskRunStatisticMap error"
					+ ", taskid:" + taskid
					+ ", runtime:" + runtime
					+ ", lastProcessResult:" + lastProcessResult
					, e);
		}

	}

	public void deleteTaskRunStatisticMap(Long taskid){
		try{
			taskRunStatisticMap.remove(taskid);
		}catch (Throwable e){
			logger.error("[LExecutorUnit]: deleteTaskRunStatisticMap error"
					+ ", taskid:" + taskid
					, e);
		}
	}

	public String getTaskRunStatisticMapStr(){
		StringBuffer taskRunStatisticStr = new StringBuffer();

		taskRunStatisticStr.append("[");
		for (TaskRunStatistic taskRunStatistic:taskRunStatisticMap.values()){
			taskRunStatisticStr.append(","+taskRunStatistic.toString());
		}
		taskRunStatisticStr.append("]");
		return taskRunStatisticStr.toString();
	}


	/**
	 * 刷新执行单元信息
	 *  executableTask
	 */
	public void refresh(ExecutableTask executableTask) {
		this.executableTask = executableTask;

		int pageSize = clientContext.getClientConfig().getPageSize();
		Map<String, Integer> pageSizeMap = clientContext.getClientConfig().getPageSizeMap();
		if(! CollectionUtils.isEmpty(pageSizeMap) && pageSizeMap.get(executableTask.getJob().getJobProcessor()) != null) {
			pageSize = clientContext.getClientConfig().checkPageSize(pageSizeMap.get(executableTask.getJob().getJobProcessor()).intValue());
		}
		this.executableTask.setLength(pageSize);

		for(int i = 0 ; i < this.longTimeTaskProcessors.length ; i ++) {
			this.longTimeTaskProcessors[i].refresh(this, i);
		}

		this.pullProcessor.refresh(this);
	}

	/**
	 * 初始化
	 *  com.alibaba.dts.common.exception.InitException
	 */
	public void init() throws InitException {
		
		/** 初始化拉任务快照线程 */
		this.pullProcessor = new PullProcessor(clientContext,this);

		this.reFillingProcessor = new ReFillingProcessor(clientContext,this);


		/** 初始化任务队列 */
		this.queue = new LinkedBlockingQueue<TaskSnapshot>(MAX_LONGTIMETASKLIST_SIZE);

		this.completedqueue = new LinkedBlockingQueue<TaskSnapshot>(MAX_LONGTIMETASKLIST_SIZE);

		this.pullProcessor.start();

		this.reFillingProcessor.start();

		initTaskProcessors();

		initStatesReportTimer();
	}

	public void initStatesReportTimer() throws InitException {
		try {

			executorService = Executors
					.newScheduledThreadPool(1, new ThreadFactory() {
						public Thread newThread(Runnable runnable) {
							return new Thread(runnable, "DTS-LongTimeTaskStates-report-thread"+"-"
									+ executableTask.getJob().getId()+"-"
									+ executableTask.getJob().getJobProcessor()+"-"
									+ executableTask.getJobInstanceSnapshot().getId()
							);
						}
					});

			executorService.scheduleAtFixedRate(new StatesReportTimer(this,clientContext),
					1 * 60 * 1000L, 2 * 60 * 1000L, TimeUnit.MILLISECONDS);
		} catch (Throwable e) {
			throw new InitException("[ExecutorUnit]: initStatesReportTimer error", e);
		}
	}

	public void activeInit() throws InitException {

		/** 初始化拉任务快照线程 */
		this.pullProcessor = new PullProcessor(clientContext,this);

		this.reFillingProcessor = new ReFillingProcessor(clientContext,this);

		/** 初始化任务队列 */
		this.queue = new LinkedBlockingQueue<TaskSnapshot>(MAX_LONGTIMETASKLIST_SIZE);

		this.completedqueue = new LinkedBlockingQueue<TaskSnapshot>(MAX_LONGTIMETASKLIST_SIZE);

		this.pullProcessor.setStop(true);

		if (this.reFillingProcessor.isStop()||(!this.reFillingProcessor.isAlive()))  {
			this.reFillingProcessor = new ReFillingProcessor(clientContext,this);
			this.reFillingProcessor.start();
		}

		initTaskProcessors();
	}

	private void initTaskProcessors() {
		try{
			int consumerThreads = this.clientContext.getClientConfig().getConsumerThreads();
			Map<String, Integer> consumerThreadsMap = this.clientContext.getClientConfig().getConsumerThreadsMap();
			if(! CollectionUtils.isEmpty(consumerThreadsMap) && consumerThreadsMap.get(executableTask.getJob().getJobProcessor()) != null) {
				consumerThreads = this.clientContext.getClientConfig().checkConsumerThreads(consumerThreadsMap.get(executableTask.getJob().getJobProcessor()).intValue());
			}

			if(executableTask.getRunThreads() > 0) {
				consumerThreads = executableTask.getRunThreads();
			}

			/** 初始化任务处理线程组 */
			this.longTimeTaskProcessors = new LongTimeTaskProcessor[consumerThreads];
			for(int i = 0 ; i < consumerThreads ; i ++) {
				this.longTimeTaskProcessors[i] = new LongTimeTaskProcessor(clientContext,this, i, this.threadCounter);
				this.longTimeTaskProcessors[i].start();
			}
		}catch (Throwable e){
			logger.error("[LExecutorUnit]: initTaskProcessors error"
					, e);
		}

	}

	public void restartPull(){

		try{
			if (!isExistsProcessors()){
				initTaskProcessors();
			}

			if (this.pullProcessor==null||this.pullProcessor.isStop()||(!this.pullProcessor.isAlive()))  {
				this.pullProcessor = new PullProcessor(clientContext,this);
				this.pullProcessor.start();
			}

			logger.info("[LExecutorUnit]: restartPull start!");

		}catch (Throwable e){
			logger.error("[LExecutorUnit]: restartPull error"
					, e);
		}
	}

	private boolean isExistsProcessors(){
		Boolean result=false;
		if (this.longTimeTaskProcessors == null){
			return false;
		}else{
			for(int i = 0 ; i < this.longTimeTaskProcessors.length ; i ++) {
				if ((this.longTimeTaskProcessors[i]!=null)||(!this.longTimeTaskProcessors[i].isStop())||(this.longTimeTaskProcessors[i].isAlive()))  {
					return true;
				}
			}
		}
		return result;
	}


	public void releaseCompleteTask(){

		if (!isPullTaskFlag()){

			try {
				this.setReleaseTaskFlag(true);
				BlockingQueue<TaskSnapshot> queue = this.getQueue();
				BlockingQueue<TaskSnapshot> completeQueue = this.getCompletedqueue();


				int queueTotal= queue.size();
				int completeQueueTotal = completeQueue.size();
				int releaseQueueSuccess = 0;

				while(! queue.isEmpty()) {
					try {
						TaskSnapshot taskSnapshot = queue.poll();
						if (taskSnapshot!=null)	releaseQueue.put(taskSnapshot);
					} catch (Throwable e) {
						logger.error("[LReleaseProcessor]: pullQueue error"
								+ ", instanceId:" + this.getExecutableTask().getJobInstanceSnapshot().getId(), e);
					}
				}

				while(! completeQueue.isEmpty()) {
					try {
						TaskSnapshot taskSnapshot = completeQueue.poll();
						if (taskSnapshot!=null)  releaseQueue.put(taskSnapshot);
					} catch (Throwable e) {
						logger.error("[LReleaseProcessor]: pullAndPut error"
								+ ", instanceId:" + this.getExecutableTask().getJobInstanceSnapshot().getId(), e);
					}
				}

				int releaseQueueTotal = releaseQueue.size();

				while(! releaseQueue.isEmpty()) {
					try {

						Result<Boolean> ackResult = null;

						TaskSnapshot taskSnapshot = releaseQueue.poll();

						if (taskSnapshot!=null){
							 ackResult = this.clientContext.getExecutor().acknowledgeRes(taskSnapshot, TASK_STATUS_ALLOCATION, 0);
						}

						if (ackResult!=null && ackResult.getData().booleanValue()){
							deleteTaskRunStatisticMap(taskSnapshot.getId());
							releaseQueueSuccess++;
							logger.info("[LReleaseProcessor] release task, instanceid:"+taskSnapshot.getJobInstanceId()+",taskid(db):"+taskSnapshot.getId());
						}else{
							if (taskSnapshot!=null){
								queue.put(taskSnapshot);
								logger.warn("[LReleaseProcessor] release task failur, reenter queue, instanceid:"+taskSnapshot.getJobInstanceId()+",taskid(db):"+taskSnapshot.getId() +",ackResult:"+ackResult.toString());
							}
						}

					} catch (Throwable e) {
						logger.error("[LReleaseProcessor]: pullCompleteQueue error"
								+ ", instanceId:" + this.getExecutableTask().getJobInstanceSnapshot().getId(), e);
					}
				}

				logger.info("[LReleaseProcessor]: releaseQueue end"
								+ ", queueTotal:" + queueTotal
								+ ", completeQueueTotal:" + completeQueueTotal
								+ ", releaseQueueTotal:" + releaseQueueTotal
								+ ", releaseQueueSuccess:" + releaseQueueSuccess
				);


			} catch (Throwable e) {
				logger.error("[LReleaseProcessor]: run error"
						+ ", instanceId:" + this.getExecutableTask().getJobInstanceSnapshot().getId(), e);
			} finally {
				this.setReleaseTaskFlag(false);
			}

		}
	}



	public void clear() {
		try {
			int queueTotal= queue.size();
			int completeQueueTotal = completedqueue.size();
			int releaseQueueTotal = releaseQueue.size();
			int taskRunStatisticMapTotal = taskRunStatisticMap.size();

			this.queue.clear();
			this.completedqueue.clear();
			this.releaseQueue.clear();
			this.taskRunStatisticMap.clear();

			logger.info("[LExecutorUnit]: clear success"
					+ ", queueTotal:" + queueTotal
					+ ", completeQueueTotal:" + completeQueueTotal
					+ ", releaseQueueTotal:" + releaseQueueTotal
					+ ", taskRunStatisticMapTotal:" + taskRunStatisticMapTotal);

		} catch (Throwable e) {
			logger.error("[LExecutorUnit]: clear error"
					+ ", instanceId:" + this.executableTask.getJobInstanceSnapshot().getId(), e);
		}
	}
	

	public void stopTask() {
		try{
			/** 强制停止拉取线程 */
			pullProcessor.setStop(true);

			reFillingProcessor.setStop(true);

			executorService.shutdown();

			/** 强制停止执行任务线程 */
			for(int i = 0 ; i < this.longTimeTaskProcessors.length ; i ++) {
				this.longTimeTaskProcessors[i].setStop(true);
			}
			clear();

			logger.info("[LExecutorUnit]: stopTask end");

		}catch (Throwable e) {
			logger.error("[LExecutorUnit]: stopTask error"
					+ ", instanceId:" + this.executableTask.getJobInstanceSnapshot().getId(), e);
		}

	}
	

	@SuppressWarnings("deprecation")
	public void forceStopTask() {

		try{

			try {
				pullProcessor.stop();
			} catch (Throwable e) {
				logger.error("[LExecutorUnit]: forceStopTask pullProcessor error"
						+ ", instanceId:" + this.executableTask.getJobInstanceSnapshot().getId(), e);
			}

			try {
				reFillingProcessor.stop();
			} catch (Throwable e) {
				logger.error("[LExecutorUnit]: forceStopTask pullProcessor error"
						+ ", instanceId:" + this.executableTask.getJobInstanceSnapshot().getId(), e);
			}

			//清空队列
			clear();

			executorService.shutdownNow();

			/** 强制停止执行任务线程 */
			for(int i = 0 ; i < this.longTimeTaskProcessors.length ; i ++) {
				try {
					this.longTimeTaskProcessors[i].stop();
				} catch (Throwable e) {
					logger.error("[LExecutorUnit]: forceStopTask parallelTaskProcessors error"
							+ ", instanceId:" + this.executableTask.getJobInstanceSnapshot().getId(), e);
				}
			}

			logger.info("[LExecutorUnit]: stopTask end");

		}catch (Throwable e) {
			logger.error("[LExecutorUnit]: forceStopTask error"
					+ ", instanceId:" + this.executableTask.getJobInstanceSnapshot().getId(), e);
		}
	}
	
	/**
	 * 执行器是否停止
	 *
	 */
	public boolean isExecutorStop() {
		return queue.isEmpty() && (threadCounter.get() == 0);
	}
	
	/**
	 * 将任务放入队列
	 *  taskSnapshot
	 *
	 */
	public boolean offer(TaskSnapshot taskSnapshot) {
		
		boolean result = false;
		try {
			result = queue.offer(taskSnapshot, DEFAULT_INVOKE_TIMEOUT, TimeUnit.MILLISECONDS);

			logger.info("[LExecutorUnit]: offer task"
					+ ",instanceId:"+taskSnapshot.getJobInstanceId()
					+ ",taskid:"+taskSnapshot.getId()
					+ ",result:"+result
			);

		} catch (Throwable e) {
			logger.error("[LExecutorUnit]: offer error"
					+ ", jobInstanceId:" + taskSnapshot.getJobInstanceId() 
					+ ", id:" + taskSnapshot.getId(), e);
		}
		
		return result;
	}

	public void taskPostProcess(TaskSnapshot taskSnapshot) {
		try {
			completedqueue.add(taskSnapshot);
			logger.info("[LExecutorUnit]: taskPostProcess"
							+ ",taskid:"+taskSnapshot.getId()
			);
		} catch (Throwable e) {
			logger.error("[LPullProcessor]: put error"
					+ ", instanceId:" + taskSnapshot.getJobInstanceId()
					+ ", id:" + taskSnapshot.getId(), e);
		}
	}

	public BlockingQueue<TaskSnapshot> getCompletedqueue() {
		return completedqueue;
	}

	public ExecutableTask getExecutableTask() {
		return executableTask;
	}

	public BlockingQueue<TaskSnapshot> getQueue() {
		return queue;
	}

	public LongTimeTaskProcessor[] getLongTimeTaskProcessors() {
		return longTimeTaskProcessors;
	}

	public AtomicInteger getThreadCounter() {
		return threadCounter;
	}

	public LongTimePool getLongTimePool() {
		return longTimePool;
	}

	@Override
	public String toString() {
		return "ExecutorUnit [executableTask=" + executableTask + "]";
	}


	public ReFillingProcessor getReFillingProcessor() {
		return reFillingProcessor;
	}


}

