package com.alibaba.dts.client.executor.longtime.processor;

import java.util.Date;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import com.alibaba.dts.client.executor.job.context.ClientContextImpl;
import com.alibaba.dts.client.executor.job.processor.LongTimeJobProcessor;
import com.alibaba.dts.client.executor.longtime.LongTimePool;
import com.alibaba.dts.client.executor.longtime.unit.ExecutorUnit;
import com.alibaba.dts.common.constants.Constants;
import com.alibaba.dts.common.domain.result.ProcessResult;
import com.alibaba.dts.common.domain.result.Result;
import com.alibaba.dts.common.domain.store.TaskSnapshot;
import com.alibaba.dts.common.logger.SchedulerXLoggerFactory;
import com.alibaba.dts.common.logger.innerlog.Logger;

/**
 * LongTime任务处理器
 *
 */
public class LongTimeTaskProcessor extends Thread implements Constants {

	private static final Logger logger = SchedulerXLoggerFactory.getLogger(LongTimeTaskProcessor.class);

	private static final Logger executeLogger = SchedulerXLoggerFactory.getExecuteLogger();
	
	/** 执行单元 */
	private ExecutorUnit executorUnit;
	
	/** 是否停止执行线程 */
	private volatile boolean stop = false;
	
	/** 状态 */
	private int status = TASK_PROCESSOR_STATUS_STOP;
	
	/** 线程计数器 */
	private AtomicInteger threadCounter;
	
	/** 并行计算上下文 */
	private LongTimeJobContext context;

	private final ClientContextImpl clientContext;

	public LongTimeTaskProcessor(final ClientContextImpl clientContext, ExecutorUnit executorUnit, int index, AtomicInteger threadCounter) {
		this.clientContext = clientContext;
		this.executorUnit = executorUnit;
		super.setName(TASK_THREAD_NAME + executorUnit.getExecutableTask().getJob().getId()
				+ HORIZONTAL_LINE +  executorUnit.getExecutableTask().getJob().getJobProcessor()
				+ HORIZONTAL_LINE +  executorUnit.getExecutableTask().getJobInstanceSnapshot().getId()
				+ HORIZONTAL_LINE +  executorUnit.getExecutableTask().getJobInstanceSnapshot().getFireTime()
				+ HORIZONTAL_LINE +  executorUnit.getExecutableTask().getJobInstanceSnapshot().getRetryCount()
				+ HORIZONTAL_LINE + index);
		this.threadCounter = threadCounter;
		this.context = new LongTimeJobContextImpl(clientContext,this.executorUnit.getExecutableTask().getJob(),
				this.executorUnit.getExecutableTask().getJobInstanceSnapshot(),
				executorUnit.getExecutableTask().getJobInstanceSnapshot().getRetryCount());


		String[] jobProcessorProperties = executorUnit.getExecutableTask().getJob().getJobProcessor().split(COLON);
		String jobProcessor = jobProcessorProperties[POSITION_PROCESSOR].trim();

		if(this.clientContext.getClientConfig().getFailureJobProcessorMap() != null
				&& this.clientContext.getClientConfig().getFailureJobProcessorMap().get(jobProcessor) != null) {

		}

		this.context.setAvailableMachineAmount(this.executorUnit.getExecutableTask().getAvailableMachineAmount());
		this.context.setCurrentMachineNumber(this.executorUnit.getExecutableTask().getCurrentMachineNumber());
	}

	/**
	 * 刷新线程信息
	 *  executorUnit
	 *  index
	 */
	public void refresh(ExecutorUnit executorUnit, int index) {
		this.executorUnit = executorUnit;
		super.setName(TASK_THREAD_NAME + executorUnit.getExecutableTask().getJob().getId()
				+ HORIZONTAL_LINE +  executorUnit.getExecutableTask().getJob().getJobProcessor()
				+ HORIZONTAL_LINE +  executorUnit.getExecutableTask().getJobInstanceSnapshot().getId()
				+ HORIZONTAL_LINE +  executorUnit.getExecutableTask().getJobInstanceSnapshot().getFireTime()
				+ HORIZONTAL_LINE +  executorUnit.getExecutableTask().getJobInstanceSnapshot().getRetryCount()
				+ HORIZONTAL_LINE + index);

		this.context = new LongTimeJobContextImpl(clientContext,this.executorUnit.getExecutableTask().getJob(),
				this.executorUnit.getExecutableTask().getJobInstanceSnapshot(), 
				executorUnit.getExecutableTask().getJobInstanceSnapshot().getRetryCount());
		

		String[] jobProcessorProperties = executorUnit.getExecutableTask().getJob().getJobProcessor().split(COLON);
		String jobProcessor = jobProcessorProperties[POSITION_PROCESSOR].trim();
		
		if(this.clientContext.getClientConfig().getFailureJobProcessorMap() != null
				&& this.clientContext.getClientConfig().getFailureJobProcessorMap().get(jobProcessor) != null) {
			

		}
	}
	
	@Override
	public void run() {
		try {
			/** 处理器准备 */
			LongTimeJobProcessor longTimeJobProcessor = null;
			try {
				longTimeJobProcessor = this.clientContext.getJobProcessorFactory().createAndGetLongTimeJobProcessor(this.executorUnit.getExecutableTask().getJob(), false);
			} catch (Throwable e) {
				logger.error("[LongTimeTaskProcessor]: createAndGetLongTimeJobProcessor error"
						+ ", jobProcessor:" + this.executorUnit.getExecutableTask().getJob().getJobProcessor(), e);
			}
			
			BlockingQueue<TaskSnapshot> queue = this.executorUnit.getQueue();
			while(! stop ) {

				TaskSnapshot taskSnapshot = null;
				try {
					taskSnapshot = queue.poll(DEFAULT_POLL_TIMEOUT, TimeUnit.MILLISECONDS);
				} catch (Throwable e) {
					logger.error("[LongTimeTaskProcessor]: take executableTask error"
							+ ", instanceId:" + this.executorUnit.getExecutableTask().getJobInstanceSnapshot().getId(), e);
				}

				if(null == taskSnapshot) {
					continue ;
				}

				/** 执行任务 */
				executeTask(taskSnapshot, longTimeJobProcessor);
			}
		} catch (Throwable e) {
			logger.error("[LongTimeTaskProcessor]: run error"
					+ ", instanceId:" + this.executorUnit.getExecutableTask().getJobInstanceSnapshot().getId(), e);
		} finally {
			try {
				LongTimePool longTimePool = executorUnit.getLongTimePool();
				longTimePool.stopTask(this.executorUnit.getExecutableTask().getJob().getId(),
						this.executorUnit.getExecutableTask().getJobInstanceSnapshot().getId());
			} catch (Throwable e) {
				logger.error("[LongTimeTaskProcessor]: finally stopTask error"
						+ ", instanceId:" + this.executorUnit.getExecutableTask().getJobInstanceSnapshot().getId(), e);
			} finally {
				if(this.clientContext.getClientConfig().isFinishLog()) {
					logger.warn("[LongTimeTaskProcessor]: finally stopTask"
							+ ", instanceId:" + this.executorUnit.getExecutableTask().getJobInstanceSnapshot().getId());
				}
			}
		}
	}

	/**
	 * 执行任务
	 *  taskSnapshot
	 *  longTimeJobProcessor
	 */
	private void executeTask(TaskSnapshot taskSnapshot, LongTimeJobProcessor longTimeJobProcessor) {

		if(null == longTimeJobProcessor) {
			logger.error("[LongTimeTaskProcessor]: jobProcessor is null"
					+ ", please check " + this.executorUnit.getExecutableTask().getJob().getJobProcessor());
			
			//失败确认
			this.clientContext.getExecutor().acknowledge(taskSnapshot, TASK_STATUS_FAILURE, 0);
			return ;
		}
		long startTime=0;
		long runtime=0;
		/** 任务开始 计数器加一 */
		this.status = TASK_PROCESSOR_STATUS_RUNNING;
		this.threadCounter.incrementAndGet();
		ProcessResult processResult = null;
		try {
			/** 设置任务 */
			this.context.setTask(taskSnapshot);
			try {
				handleTaskBefore(taskSnapshot);
				startTime= System.currentTimeMillis();
				executeLogger.info("longTimeJobProcessor start..."
								+ ",instanceId:" + taskSnapshot.getJobInstanceId()
								+ ",taskId:" + taskSnapshot.getId()
				);
				processResult = longTimeJobProcessor.process(this.context);
				runtime= System.currentTimeMillis()-startTime;
				executeLogger.info("longTimeJobProcessor end..."
								+ ",instanceId:" + taskSnapshot.getJobInstanceId()
								+ ",taskId:" + taskSnapshot.getId()
								+ ",runtime:" + runtime
								+ ",processResult:" + processResult.toString()
				);
			} catch (Throwable e) {
				
				logger.error("[LongTimeTaskProcessor]: process error"
						+ ", instanceId:" + taskSnapshot.getJobInstanceId() 
						+ ", id:" + taskSnapshot.getId(), e);
				

			}

			if(null == processResult) {
				
				logger.error("[LongTimeTaskProcessor]: process error, processResult is null"
						+ ", instanceId:" + taskSnapshot.getJobInstanceId() 
						+ ", id:" + taskSnapshot.getId());
				processResult = new ProcessResult(false);

			}

		} catch (Throwable e) {
			logger.error("[LongTimeTaskProcessor]: executeTask error"
					+ ", instanceId:" + taskSnapshot.getJobInstanceId() 
					+ ", id:" + taskSnapshot.getId(), e);
		} finally {

			handleTaskPost(taskSnapshot,processResult,runtime);
			/** 任务结束 计数器减一 */
			this.threadCounter.decrementAndGet();
			this.status = TASK_PROCESSOR_STATUS_STOP;
		}
	}

	private void handleTaskBefore(TaskSnapshot taskSnapshot) {
		try{
			if (!executorUnit.isExistsInTaskRunStatisticMap(taskSnapshot.getId())){
				executorUnit.addTaskRunStatisticMap(taskSnapshot.getId(), TASK_STATUS_START);
			}
		} catch (Throwable e) {
			logger.error("[LongTimeTaskProcessor]:  handleTaskBefore process error, thread:" + Thread.currentThread().getName(), e);
		}
	}

	/**
	 * 任务后置处理
	 *  processResult
	 */
	private void handleTaskPost(TaskSnapshot taskSnapshot,ProcessResult processResult,long runtime) {

		Long sleeptime=50L;


		if(processResult!=null && processResult.getSleepTime()>=0) {

			sleeptime= Long.valueOf(processResult.getSleepTime());

			try {
				Thread.sleep(sleeptime);
			} catch (Throwable e) {
				logger.error("[LongTimeTaskProcessor]:  post process error, thread:" + Thread.currentThread().getName(), e);
			}finally {
				if (!taskSnapshot.getTaskName().equals(DEFAULT_ROOT_LEVEL_TASK_NAME))
					this.executorUnit.taskPostProcess(taskSnapshot);
			}
		}else{
			try {
				Thread.sleep(sleeptime);
			} catch (Throwable e) {
				logger.error("[LongTimeTaskProcessor]:  post process error, thread:" + Thread.currentThread().getName(), e);
			}finally {
				if (!taskSnapshot.getTaskName().equals(DEFAULT_ROOT_LEVEL_TASK_NAME))
					this.executorUnit.taskPostProcess(taskSnapshot);
			}
		}

		try{
			Result<Boolean> ackResult=null;

			if (executorUnit.isExistsInTaskRunStatisticMap(taskSnapshot.getId())){

				if (executorUnit.getTaskRunStatisticMap().get(taskSnapshot.getId()).getProcessResult()==TASK_STATUS_START){

					if (taskSnapshot.getTaskName().equals(DEFAULT_ROOT_LEVEL_TASK_NAME)){
						ackResult = this.clientContext.getExecutor().acknowledgeRes(taskSnapshot, TASK_STATUS_SUCCESS, 0);
					} else {
						ackResult = this.clientContext.getExecutor().acknowledgeRes(taskSnapshot,TASK_STATUS_RUNNING,0);
					}

					if (processResult.isSuccess()) {

						if(null == ackResult || ! ackResult.getData().booleanValue()) {
							executorUnit.updateTaskRunStatisticMap(taskSnapshot.getId(),runtime, TASK_ACK_FAILURE);
						}else{
							executorUnit.updateTaskRunStatisticMap(taskSnapshot.getId(),runtime, TASK_STATUS_SUCCESS);
						}

					} else {
						if(null == ackResult || ! ackResult.getData().booleanValue()) {
							executorUnit.addTaskRunStatisticMap(taskSnapshot.getId(),TASK_ACK_FAILURE);
						}else{
							executorUnit.addTaskRunStatisticMap(taskSnapshot.getId(),TASK_STATUS_FAILURE);
						}

					}

				}else{

					if (processResult.isSuccess()){
						if (executorUnit.getTaskRunStatisticMap().get(taskSnapshot.getId()).getProcessResult()!=TASK_STATUS_RUNNING){
							ackResult = this.clientContext.getExecutor().acknowledgeRes(taskSnapshot, TASK_STATUS_RUNNING, 0);

							if(null == ackResult || ! ackResult.getData().booleanValue()) {
								executorUnit.updateTaskRunStatisticMap(taskSnapshot.getId(),runtime,TASK_ACK_FAILURE);
							}else{
								executorUnit.updateTaskRunStatisticMap(taskSnapshot.getId(),runtime,TASK_STATUS_RUNNING);
							}

						}else {
							executorUnit.updateTaskRunStatisticMap(taskSnapshot.getId(),runtime);
						}
					}else{
						if (executorUnit.getTaskRunStatisticMap().get(taskSnapshot.getId()).getProcessResult()!=TASK_STATUS_FAILURE){
							ackResult = this.clientContext.getExecutor().acknowledgeRes(taskSnapshot, TASK_STATUS_FAILURE, 0);

							if(null == ackResult || ! ackResult.getData().booleanValue()) {
								executorUnit.updateTaskRunStatisticMap(taskSnapshot.getId(),runtime,TASK_ACK_FAILURE);
							}else{
								executorUnit.updateTaskRunStatisticMap(taskSnapshot.getId(),runtime,TASK_STATUS_FAILURE);
							}
						}else {
							executorUnit.updateTaskRunStatisticMap(taskSnapshot.getId(), runtime);
						}
					}

				}

			}else{

				logger.error("[LongTimeTaskProcessor]:  updateTaskRunStatisticMap error, task not exist,taskid:" + taskSnapshot.getId());

			}
		} catch (Throwable e) {
			logger.error("[LongTimeTaskProcessor]:  acknowledge process error, thread:" + Thread.currentThread().getName(), e);
		}

	}
	

	public boolean isStop() {
		return stop;
	}

	public void setStop(boolean stop) {
		this.stop = stop;
		this.context.setStop(stop);
	}

	public int getStatus() {
		return status;
	}

}
