package com.alibaba.dts.client.executor.logcollector;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.io.Writer;
import java.util.ArrayList;
import java.util.List;

import com.alibaba.dts.common.logger.SchedulerXLoggerFactory;
import com.alibaba.dts.common.logger.innerlog.Logger;

/**
 *
 * @author xiaomeng.hxm
 */
public class LocalLogCollector extends AbstractLogCollector {
    private static final Logger LOGGER = SchedulerXLoggerFactory.getLogger(LocalLogCollector.class);

    private String fileName;
    private String fireDate;
    public static final String DIR_PREFIX = System.getProperties().getProperty("user.home") + "/logs/schedulerx-log/script";

    public LocalLogCollector(String fireDate, String fileName) {
        this.fireDate = fireDate;
        this.fileName = fileName;
    }

    private void write(String text, StreamType streamType) {
        String type = (streamType == StreamType.STD_OUT) ? ".out" : ".err";
        String dirPath = DIR_PREFIX + "/" + fireDate + "/";
        File dir = new File(dirPath);
        if (!dir.exists()) {
            dir.mkdirs();
        }
        String logFile = dirPath + fileName + type;
        Writer writer = null;
        try {
            writer = new FileWriter(logFile, true);
            writer.write(text);
        } catch (IOException e) {
            LOGGER.error("", e);
        } finally {
            if (writer != null) {
                try {
                    writer.close();
                } catch (IOException e) {
                    LOGGER.error("", e);
                }
            }
        }
    }


    @Override
    public void collect(String line, StreamType streamType, boolean isEnd) {
        if (isEnd) {
            write(String.valueOf(AbstractLogCollector.END_OF_LOG), streamType);
        } else {
            write(line + "\n", streamType);
        }

    }

    @Override
    public List<String> readLines(long lineOffset, int lineNumber, StreamType streamType) throws IOException {
        List<String> lines = new ArrayList<String>();

        if (lineNumber <= 0 || lineOffset < 0) {
            return lines;
        }

        String type = (streamType == StreamType.STD_OUT) ? ".out" : ".err";
        String logFile = DIR_PREFIX + "/" + fireDate + "/" + fileName + type;
        FileReader reader = null;
        BufferedReader br = null;
        String endOfLog = String.valueOf(AbstractLogCollector.END_OF_LOG);
        try {
            reader = new FileReader(logFile);
            br = new BufferedReader(reader);
            String line = null;
            int offsetCounter = 0;

            //跳过指定行
            while (offsetCounter < lineOffset) {
                line = br.readLine();
                if (line == null || line.equals(endOfLog)) {
                    break;
                }
                offsetCounter++;
            }

            //开始读若干行
            int lineCounter = 0;
            while (lineCounter < lineNumber) {
                line = br.readLine();
                if (line == null || line.equals(endOfLog)) {
                    break;
                }
                lineCounter++;
                lines.add(line);
            }
        } catch (IOException e) {
            LOGGER.error("", e);
            throw e;
        } finally {
            try {
                if (br != null) {
                    br.close();
                }
                if (reader != null) {
                    reader.close();
                }
            } catch (Exception e) {
                LOGGER.error("", e);
            }
        }

        return lines;
    }

    @Override
    public List<String> tailLines(int lineNumber, StreamType streamType) throws IOException {
        List<String> lines = new ArrayList<String>();
        String type = (streamType == StreamType.STD_OUT) ? ".out" : ".err";
        String logFile = DIR_PREFIX + "/" + fireDate + "/" + fileName + type;
        RandomAccessFile rf = null;
        int counter=0;
        try {
            rf = new RandomAccessFile(logFile, "r");
            long len = rf.length();
            long start = rf.getFilePointer();
            long nextend = start + len - 1;
            String line;
            rf.seek(nextend);
            int c = -1;
            while (nextend > start && counter < lineNumber) {
                c = rf.read();
                if (c == AbstractLogCollector.END_OF_LOG) {
                    nextend--;
                    continue;
                }
                if (c == '\n' || c == '\r') {
                    line = rf.readLine();
                    if (line != null) {
                        lines.add(line);
                        counter++;
                    }
                    nextend--;
                }
                nextend--;
                rf.seek(nextend);
                if (nextend == 0) {// 当文件指针退至文件开始处，输出第一行
                    counter++;
                    lines.add(rf.readLine());
                }
              }
        } catch (IOException e) {
            LOGGER.error("", e);
            throw e;
        } finally {
            try {
                if (rf != null)
                    rf.close();
                } catch (IOException e) {
                    LOGGER.error("", e);
                }
        }

        return lines;
    }

    @Override
    public void delete() throws IOException {
        File outFile = new File(DIR_PREFIX + "/" + fireDate + "/" + fileName + ".out");
        if (outFile.exists()) {
            outFile.delete();
        }

        File errFile = new File(DIR_PREFIX + "/" + fireDate + "/" + fileName + ".err");
        if (errFile.exists()) {
            errFile.delete();
        }

    }
}
