package com.alibaba.dts.client.executor.grid.timer;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import com.alibaba.dts.client.executor.job.context.ClientContextImpl;
import com.alibaba.dts.client.store.access.ExecutionCounterDao;
import com.alibaba.dts.common.domain.store.ExecutionCounter;
import com.alibaba.dts.common.exception.AccessException;
import com.alibaba.dts.common.exception.InitException;
import com.alibaba.dts.common.logger.SchedulerXLoggerFactory;
import com.alibaba.dts.common.logger.innerlog.Logger;

/**
 * ExecutionCounter persistence time, used for initialing execution counters stored in db, starting the update timer
 * <p>
 * Created by yif on 2016/10/18.
 */
public class ExecutionCounterUpdateTimer {
    private static final Logger logger = SchedulerXLoggerFactory.getLogger(ExecutionCounterUpdateTimer.class);

    private ClientContextImpl clientContext;

    private ExecutionCounterDao executionCounterDao;

    public ExecutionCounterUpdateTimer(ClientContextImpl clientContext) {
        this.clientContext = clientContext;
        executionCounterDao = clientContext.getStore().getExecutionCounterDao();
    }


    public void init() throws InitException {

        final ConcurrentHashMap<Long, ConcurrentHashMap<String, ConcurrentHashMap<String, ExecutionCounter>>> executionCounterTable = clientContext.getExecutionCounterTable();

        try {
            loadExecutionCounters(executionCounterTable);

            Executors.newScheduledThreadPool(1).scheduleAtFixedRate(new Runnable() {
                @Override
                public void run() {
                    try {

                        for (Map.Entry<Long, ConcurrentHashMap<String, ConcurrentHashMap<String, ExecutionCounter>>> executionCounterEntry : executionCounterTable.entrySet()) {
                            Long jobInstanceId = executionCounterEntry.getKey();
                            ConcurrentHashMap<String, ConcurrentHashMap<String, ExecutionCounter>> executionCounterMapByReceiveNode = executionCounterEntry.getValue();
                            for (Map.Entry<String, ConcurrentHashMap<String, ExecutionCounter>> entryByReceiveNode : executionCounterMapByReceiveNode.entrySet()) {
                                String receiveNode = entryByReceiveNode.getKey();
                                ConcurrentHashMap<String, ExecutionCounter> executionCounterMapByTaskName = entryByReceiveNode.getValue();

                                for (Map.Entry<String, ExecutionCounter> entryByTaskName : executionCounterMapByTaskName.entrySet()) {
                                    String taskName = entryByTaskName.getKey();
                                    ExecutionCounter executionCounter = entryByTaskName.getValue();
                                    executionCounter.setJobInstanceId(jobInstanceId);
                                    executionCounter.setReceiveNode(receiveNode);
                                    executionCounter.setTaskName(taskName);
                                    ExecutionCounter executionCounterDb = executionCounterDao.getByJobInstanceAndExecutionNodeAndTaskName(jobInstanceId, receiveNode, taskName);
                                    if (executionCounterDb == null) {
                                        executionCounterDao.createExecutionCounter(executionCounter);
                                    } else {
                                        if (executionCounter.getId() == 0) {
                                            executionCounter.setId(executionCounterDb.getId());
                                        }
                                        executionCounterDao.updateExecutionCounter(executionCounter);
                                    }
                                }
                            }
                        }
                    } catch (Throwable throwable) {
                        logger.error(throwable.getMessage(), throwable);
                    }
                }
            }, 10, 1, TimeUnit.SECONDS);
        } catch (Throwable throwable) {
            logger.error("ExecutionCounterUpdateTimer init error");
            throw new InitException("ExecutionCounterUpdateTimer init error", throwable);
        }
    }

    /**
     * load execution counter from db
     *
     *  executionCounterTable
     */
    private void loadExecutionCounters(ConcurrentHashMap<Long, ConcurrentHashMap<String, ConcurrentHashMap<String, ExecutionCounter>>> executionCounterTable) {
        List<ExecutionCounter> executionCounters = null;
        try {
            executionCounters = executionCounterDao.list();
        } catch (AccessException e) {
            logger.error("list executions failed", e);
        }
        if (executionCounters == null || executionCounters.isEmpty()) {
            return;
        }

        for (ExecutionCounter executionCounter : executionCounters) {
            Long jobInstanceId = executionCounter.getJobInstanceId();
            String receiveNode = executionCounter.getReceiveNode();
            String taskName = executionCounter.getTaskName();
            ConcurrentHashMap<String, ConcurrentHashMap<String, ExecutionCounter>> executionCounterMapByReceiveNode = executionCounterTable.get(jobInstanceId);
            if (executionCounterMapByReceiveNode == null) {
                executionCounterMapByReceiveNode = new ConcurrentHashMap<String, ConcurrentHashMap<String, ExecutionCounter>>();
                ConcurrentHashMap<String, ConcurrentHashMap<String, ExecutionCounter>> executionCounterMapByReceiveNodeExist = clientContext.getExecutionCounterTable().putIfAbsent(jobInstanceId, executionCounterMapByReceiveNode);
                if (executionCounterMapByReceiveNodeExist != null) {
                    executionCounterMapByReceiveNode = executionCounterMapByReceiveNodeExist;
                }
            }

            ConcurrentHashMap<String, ExecutionCounter> executionCounterMapByTaskName = executionCounterMapByReceiveNode.get(receiveNode);
            if (executionCounterMapByTaskName == null) {
                executionCounterMapByTaskName = new ConcurrentHashMap<String, ExecutionCounter>();
                ConcurrentHashMap<String, ExecutionCounter> executionCounterMapByTaskNameExist = executionCounterMapByReceiveNode.putIfAbsent(receiveNode, executionCounterMapByTaskName);
                if (executionCounterMapByTaskNameExist != null) {
                    executionCounterMapByTaskName = executionCounterMapByTaskNameExist;
                }
            }

            executionCounterMapByTaskName.putIfAbsent(taskName, executionCounter);
        }

    }


}
