package com.alibaba.dts.client.executor.grid.queue.send;

import java.util.Date;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import com.alibaba.dts.client.executor.grid.queue.TaskEvent;
import com.alibaba.dts.client.executor.job.context.ClientContextImpl;
import com.alibaba.dts.common.domain.remoting.RemoteMachine;
import com.alibaba.dts.common.exception.InitException;
import com.alibaba.dts.common.logger.SchedulerXLoggerFactory;
import com.alibaba.dts.common.logger.innerlog.Logger;
import com.alibaba.dts.common.util.NamedThreadFactory;

/**
 * Created by yif on 2017/4/10.
 */
public class SendManager {
    private static final Logger logger = SchedulerXLoggerFactory.getLogger(SendManager.class);

    private AtomicBoolean readyForSend = new AtomicBoolean(false);
    private ClientContextImpl clientContext;

    private int routeQueueSize = 8 * 1024 * 1024;
    private int mergeQueueSize = 8 * 1024 * 1024;
    private int sendQueueSize = 8 * 1024 * 1024;

    private int routeThreadCount = 4;
    private int mergeThreadCount = 1;
    private int sendThreadCount = 16;

    private BlockingQueue<TaskEvent> routeQueue;
    private BlockingQueue<TaskEvent> mergeQueue;
    private BlockingQueue<TaskEvent> sendQueue;

    private ConcurrentHashMap<String, ConcurrentHashMap<Long, MergingTaskGroup>>
        mergingTaskGroupMapByTargetMachine
        = new ConcurrentHashMap<String, ConcurrentHashMap<Long, MergingTaskGroup>>();

    private ConcurrentHashMap<String, ConcurrentHashMap<Long, MergingTaskGroup>>
        mergingTaskGroupMapByTargetMachineCompensation
        = new ConcurrentHashMap<String, ConcurrentHashMap<Long, MergingTaskGroup>>();

    private ConcurrentHashMap<Long, Object> interruptedJobInstanceMap = new ConcurrentHashMap<Long, Object>();

    private ConcurrentHashMap<Long, List<RemoteMachine>> machinesByJob = new ConcurrentHashMap<Long,
        List<RemoteMachine>>();

    public void init(ClientContextImpl clientContext) throws InitException {
        try {
            this.clientContext = clientContext;
            routeQueue = new ArrayBlockingQueue<TaskEvent>(routeQueueSize);
            mergeQueue = new ArrayBlockingQueue<TaskEvent>(mergeQueueSize);
            sendQueue = new ArrayBlockingQueue<TaskEvent>(sendQueueSize);

            ExecutorService routeThreadPool = Executors.newFixedThreadPool(routeThreadCount,
                new NamedThreadFactory("SchedulerX-Task-Route-"));
            for (int i = 0; i < routeThreadCount; i++) {
                TaskRouter taskRouter = new TaskRouter(clientContext, this);
                routeThreadPool.submit(taskRouter);
            }

            ExecutorService mergeThreadPool = Executors.newFixedThreadPool(mergeThreadCount,
                new NamedThreadFactory("SchedulerX-Task-Merge-"));
            for (int i = 0; i < mergeThreadCount; i++) {
                mergeThreadPool.submit(new TaskMerger(this));
            }

            ExecutorService sendThreadPool = Executors.newFixedThreadPool(sendThreadCount,
                new NamedThreadFactory("SchedulerX-Task-Send-"));
            for (int i = 0; i < sendThreadCount; i++) {
                sendThreadPool.submit(new TaskSender(clientContext, this));
            }

            Executors.newScheduledThreadPool(1, new NamedThreadFactory("SchedulerX-TaskMergeMonitor -"))
                .scheduleAtFixedRate(new TaskMergeMonitor(this), 0, 2,
                    TimeUnit.SECONDS);
        } catch (Throwable throwable) {
            throw new InitException("failed to init SendManager", throwable);
        }
    }

    public void putTasksToRouteQueue(List<TaskEvent> taskEvents, long jobInstanceId) {
        if (isInterruptedJobInstance(jobInstanceId)) {
            return;
        }
        for (TaskEvent taskEvent : taskEvents) {
            putSingleTaskToRouteQueue(taskEvent);
        }
    }

    public void putSingleTaskToRouteQueue(TaskEvent taskEvent) {
        try {
            routeQueue.put(taskEvent);
        } catch (InterruptedException e) {

        }
    }

    /**
     * update the job machines cache
     *
     *  jobId
     *  machines
     */
    public void resetRoutesMachines(long jobId, List<RemoteMachine> machines) {
        machinesByJob.put(jobId, machines);
    }

    public boolean isInterruptedJobInstance(long jobInstanceId) {
        return interruptedJobInstanceMap.containsKey(jobInstanceId);
    }

    public void addInterruptedInstance(long instanceId) {
        try {
            interruptedJobInstanceMap.put(instanceId, new Date());
            logger.info("[SendManager]: addInterruptedInstance instanceId:" + instanceId);
        } catch (Throwable e) {
            logger.error("[SendManager]: addInterruptedInstance error,instanceId:" + instanceId, e);
        }
    }

    public void removeInterruptedJobInstance(long instanceId) {
        try {
            interruptedJobInstanceMap.remove(instanceId);
            logger.info("[SendManager]: removeInterceptInstance instanceId:" + instanceId);
        } catch (Throwable e) {
            logger.error("[SendManager]: removeInterceptInstance error,instanceId:" + instanceId, e);
        }
    }

    public void clearMergingTaskGroupMap(long jobInstanceId) {
        try {
            for (ConcurrentHashMap<Long, MergingTaskGroup> remoteMachineMapByJobInstanceId :
                mergingTaskGroupMapByTargetMachine
                    .values()) {
                remoteMachineMapByJobInstanceId.remove(jobInstanceId);
            }
            for (ConcurrentHashMap<Long, MergingTaskGroup> remoteMachineMapByJobInstanceId :
                mergingTaskGroupMapByTargetMachineCompensation
                    .values()) {
                remoteMachineMapByJobInstanceId.remove(jobInstanceId);
            }
        } catch (Throwable throwable) {
            logger.error("faild to clearMergingTaskGroupMap, jobInstanceId=" + jobInstanceId, throwable);
        }

    }

    public BlockingQueue<TaskEvent> getRouteQueue() {
        return routeQueue;
    }

    public BlockingQueue<TaskEvent> getMergeQueue() {
        return mergeQueue;
    }

    public BlockingQueue<TaskEvent> getSendQueue() {
        return sendQueue;
    }

    public ConcurrentHashMap<Long, List<RemoteMachine>> getMachinesByJob() {
        return machinesByJob;
    }

    public ConcurrentHashMap<Long, Object> getInterruptedJobInstanceMap() {
        return interruptedJobInstanceMap;
    }

    public ConcurrentHashMap<String, ConcurrentHashMap<Long, MergingTaskGroup>>
    getMergingTaskGroupMapByTargetMachine() {
        return mergingTaskGroupMapByTargetMachine;
    }

    public ConcurrentHashMap<String, ConcurrentHashMap<Long, MergingTaskGroup>>
    getMergingTaskGroupMapByTargetMachineCompensation() {
        return mergingTaskGroupMapByTargetMachineCompensation;
    }

    public AtomicBoolean getReadyForSend() {
        return readyForSend;
    }

    public ClientContextImpl getClientContext() {
        return clientContext;
    }

    public void setRouteQueueSize(int routeQueueSize) {
        this.routeQueueSize = routeQueueSize;
    }

    public void setMergeQueueSize(int mergeQueueSize) {
        this.mergeQueueSize = mergeQueueSize;
    }

    public void setSendQueueSize(int sendQueueSize) {
        this.sendQueueSize = sendQueueSize;
    }
}
