package com.alibaba.dts.client.executor;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;

import org.springframework.util.CollectionUtils;

import com.alibaba.dts.client.executor.grid.GridPool;
import com.alibaba.dts.client.executor.grid.queue.TaskEvent;
import com.alibaba.dts.client.executor.job.context.ClientContextImpl;
import com.alibaba.dts.client.executor.logcollector.AbstractLogCollector;
import com.alibaba.dts.client.executor.logcollector.LogCollectorFactory;
import com.alibaba.dts.client.executor.logcollector.StreamType;
import com.alibaba.dts.client.executor.longtime.LongTimePool;
import com.alibaba.dts.client.executor.parallel.ParallelPool;
import com.alibaba.dts.client.executor.simple.ScxSimplePool;
import com.alibaba.dts.client.executor.simple.SimplePool;
import com.alibaba.dts.client.executor.stop.StopJob;
import com.alibaba.dts.common.constants.Constants;
import com.alibaba.dts.common.context.InvocationContext;
import com.alibaba.dts.common.domain.ExecutableTask;
import com.alibaba.dts.common.domain.remoting.RemoteMachine;
import com.alibaba.dts.common.domain.result.Result;
import com.alibaba.dts.common.domain.result.ResultCode;
import com.alibaba.dts.common.domain.store.ExecutionCounter;
import com.alibaba.dts.common.domain.store.JobInstanceSnapshot;
import com.alibaba.dts.common.domain.store.TaskSnapshot;
import com.alibaba.dts.common.exception.AccessException;
import com.alibaba.dts.common.logger.SchedulerXLoggerFactory;
import com.alibaba.dts.common.logger.innerlog.Logger;
import com.alibaba.dts.common.service.NodeServerService;
import com.alibaba.dts.common.service.ServerService;
import com.alibaba.dts.common.util.CommonUtil;
import com.alibaba.dts.common.util.StringUtil;

/**
 * 执行任务容器
 *
 * @author tianyao.myc
 */
public class Executor implements Constants {

    private static final Logger logger = SchedulerXLoggerFactory.getLogger(Executor.class);
    /**
     * 简单job执行池
     */
    private final SimplePool simplePool;

    /**
     * scx简单job执行池
     */
    private final ScxSimplePool scxSimplePool;

    /**
     * 并行计算job执行池
     */

    private ParallelPool parallelPool;

    /**
     * 网格Job执行池
     */
    private GridPool gridPool;

    /**
     * LongTimejob执行池
     */
    private LongTimePool longTimePool;

    private final StopJob stopJob;

    private ServerService serverService;

    private NodeServerService nodeServerService;

    private final ClientContextImpl clientContext;

    public Executor(final ClientContextImpl clientContext) {
        this.clientContext = clientContext;

        this.stopJob = new StopJob(this.clientContext);
        this.simplePool = new SimplePool(this.clientContext);
        this.scxSimplePool = new ScxSimplePool(this.clientContext);
        this.parallelPool = new ParallelPool(this.clientContext);
        this.serverService = this.clientContext.getClientRemoting().proxyInterface(ServerService.class);
        nodeServerService = clientContext.getNodeRemoting().proxyServerInterface();
        this.longTimePool = new LongTimePool(this.clientContext);
        this.gridPool = new GridPool(this.clientContext);

    }


    public Result<Boolean> executeTask(ExecutableTask executableTask) {
        logger.warn("[Executor]: executeTask start, jobid:" + executableTask.getJob().getId() +
                ",jobInstanceId:" + executableTask.getJobInstanceSnapshot().getId() +
                ",invokeSource:" + executableTask.getSource()
        );
        long startime = System.currentTimeMillis();
        Result<Boolean> result = new Result<Boolean>(false);
        boolean executeResult = false;
        try{

            int jobType = executableTask.getJob().getType();
            if (CommonUtil.isSimpleJob(jobType)) {

                if (Constants.ENVIRONMENT_SCX.equals(this.clientContext.getClientConfig().getEnvironment()) && !this.clientContext.getClientConfig().isNewVersion()) {
                    executeResult = scxSimplePool.executeTask(executableTask);
                } else {
                    executeResult = simplePool.executeTask(executableTask);
                }

            } else if (CommonUtil.isLongTimeJob(jobType)) {
                executeResult = longTimePool.executeTask(executableTask);
            } else if (CommonUtil.isParallelJob(jobType)) {
                executeResult = parallelPool.executeTask(executableTask);
            } else if (CommonUtil.isGridJob(jobType)) {
                executeResult = gridPool.executeTask(executableTask);
            }
            result.setData(executeResult);
            result.setResultCode(executeResult ? ResultCode.SUCCESS : ResultCode.FAILURE);

            long triggerTime = System.currentTimeMillis() - startime;
            logger.warn("[Executor]: executeTask finished, jobid:" + executableTask.getJob().getId() +
                    ",jobInstanceId:" + executableTask.getJobInstanceSnapshot().getId() +
                    ",result:" + executeResult +
                    ",elapsed time:"  + triggerTime +
                    ",invokeSource:" + executableTask.getSource()
            );
        }catch(Exception e){
            logger.error("[Executor]: executeTask error , jobid:" + executableTask.getJob().getId() +
                    ",jobInstanceId:" + executableTask.getJobInstanceSnapshot().getId() +
                    ",invokeSource:" + executableTask.getSource()  ,e);
        }
        return result;
    }

    public Result<Boolean> activeTask(ExecutableTask executableTask) {
        Result<Boolean> result = new Result<Boolean>(false);
        boolean executeResult = false;

        if (CommonUtil.isLongTimeJob(executableTask.getJob().getType())) {
            executeResult = longTimePool.activeTask(executableTask);
        }
        result.setData(executeResult);
        result.setResultCode(executeResult ? ResultCode.SUCCESS : ResultCode.FAILURE);
        return result;
    }

    public Result<Boolean> releaseCompleteTask(ExecutableTask executableTask) {
        Result<Boolean> result = new Result<Boolean>(false);
        boolean executeResult = false;

        if (CommonUtil.isLongTimeJob(executableTask.getJob().getType())) {
            executeResult = longTimePool.releaseCompleteTask(executableTask);
        }

        result.setData(executeResult);
        result.setResultCode(executeResult ? ResultCode.SUCCESS : ResultCode.FAILURE);
        return result;
    }


    public Result<Boolean> stopTask(int jobType, long jobId, long jobInstanceId) {
        Result<Boolean> result = new Result<Boolean>(false);

        boolean stopResult = false;

        if (CommonUtil.isSimpleJob(jobType)) {

            if (Constants.ENVIRONMENT_SCX.equals(this.clientContext.getClientConfig().getEnvironment()) && !this.clientContext.getClientConfig().isNewVersion()) {
                stopResult = scxSimplePool.stopTask(jobId, jobInstanceId);
            } else {
                stopResult = simplePool.stopTask(jobId, jobInstanceId);
            }

        } else if (CommonUtil.isLongTimeJob(jobType)) {
            stopResult = longTimePool.stopTask(jobId, jobInstanceId);
        } else if (CommonUtil.isParallelJob(jobType)) {
            stopResult = parallelPool.stopTask(jobId, jobInstanceId);
        } else if (CommonUtil.isGridJob(jobType)) {
            stopResult = gridPool.stopTask(jobId, jobInstanceId);
        }

        //停止任务
        this.stopJob.stopTask(jobId, jobInstanceId);

        result.setData(stopResult);
        result.setResultCode(stopResult ? ResultCode.SUCCESS : ResultCode.FAILURE);
        return result;
    }

    public void stopService(){
        simplePool.stopService();
        scxSimplePool.stopService();
        parallelPool.stopService();
        gridPool.stopService();
        longTimePool.stopService();
    }


    public Result<Boolean> forceStopTask(ExecutableTask executableTask) {
        Result<Boolean> result = new Result<Boolean>(false);

        boolean stopResult = false;
        if (CommonUtil.isSimpleJob(executableTask.getJob().getType())) {
            if (Constants.ENVIRONMENT_SCX.equals(this.clientContext.getClientConfig().getEnvironment()) && !this.clientContext.getClientConfig().isNewVersion()) {
                stopResult = scxSimplePool.forceStopTask(executableTask.getJob().getId(), executableTask.getJobInstanceSnapshot().getId());
            } else {
                stopResult = simplePool.forceStopTask(executableTask.getJob().getId(), executableTask.getJobInstanceSnapshot().getId());
            }
        } else {
            if (CommonUtil.isLongTimeJob(executableTask.getJob().getType())) {
                stopResult = longTimePool.forceStopTask(executableTask.getJob().getId(), executableTask.getJobInstanceSnapshot().getId());
            } else {
                stopResult = parallelPool.forceStopTask(executableTask.getJob().getId(), executableTask.getJobInstanceSnapshot().getId());
            }
        }

        //停止任务
        this.stopJob.stopTask(executableTask.getJob().getId(), executableTask.getJobInstanceSnapshot().getId());

        result.setData(stopResult);
        result.setResultCode(stopResult ? ResultCode.SUCCESS : ResultCode.FAILURE);
        return result;
    }


    public Result<ExecutableTask> pull(ExecutableTask executableTask) {
        List<String> serverList = this.clientContext.getClientRemoting().getServerList();
        if (CollectionUtils.isEmpty(serverList)) {
            logger.error("[Executor]: pull serverList is isEmpty error"
                    + ", instanceId:" + executableTask.getJobInstanceSnapshot().getId());
            return null;
        }

        /** 随机列表顺序 */
        Collections.shuffle(serverList);

        Result<ExecutableTask> result = null;
        for (String server : serverList) {

            try {
                InvocationContext.setRemoteMachine(new RemoteMachine(server, 12 * DEFAULT_INVOKE_TIMEOUT));
                result = serverService.pull(executableTask);
            } catch (Throwable e) {
                logger.error("[Executor]: serverService pull error"
                        + ", instanceId:" + executableTask.getJobInstanceSnapshot().getId()
                        + ", server:" + server, e);
            }

            if (null == result) {
                logger.error("[Executor]: serverService pull failed"
                        + ", instanceId:" + executableTask.getJobInstanceSnapshot().getId()
                        + ", server:" + server);
            } else {
                return result;
            }

        }

        return result;
    }


    public Result<ExecutableTask> pullLongTimeTask(ExecutableTask executableTask) {
        List<String> serverList = this.clientContext.getClientRemoting().getServerList();
        if (CollectionUtils.isEmpty(serverList)) {
            logger.error("[Executor]: pull serverList is isEmpty error"
                    + ", instanceId:" + executableTask.getJobInstanceSnapshot().getId());
            return null;
        }

        /** 随机列表顺序 */
        Collections.shuffle(serverList);

        Result<ExecutableTask> result = null;
        for (String server : serverList) {

            try {
                InvocationContext.setRemoteMachine(new RemoteMachine(server, 12 * DEFAULT_INVOKE_TIMEOUT));
                result = serverService.pullLongTimeTask(executableTask);
            } catch (Throwable e) {
                logger.error("[Executor]: serverService pullLongTimeTask error"
                        + ", instanceId:" + executableTask.getJobInstanceSnapshot().getId()
                        + ", server:" + server, e);
            }

            if (null == result) {
                logger.error("[Executor]: serverService pullLongTimeTask failed"
                        + ", instanceId:" + executableTask.getJobInstanceSnapshot().getId()
                        + ", server:" + server);
            } else {
                return result;
            }

        }

        return result;
    }


    public void acknowledge(TaskSnapshot taskSnapshot, int status, int retryTimes) {
        List<String> serverList = this.clientContext.getClientRemoting().getServerList();
        if (CollectionUtils.isEmpty(serverList)) {
            logger.error("[Executor]: acknowledge serverList is isEmpty error"
                    + ", status:" + status
                    + ", retryTimes:" + retryTimes
                    + ", instanceId:" + taskSnapshot.getJobInstanceId()
                    + ", id:" + taskSnapshot.getId());
            return;
        }

        /** 随机列表顺序 */
        Collections.shuffle(serverList);

        //如果clientId为空就补上
        if (StringUtil.isBlank(taskSnapshot.getClientId())) {
            taskSnapshot.setClientId(this.clientContext.getClientConfig().getClientId());
        }

        taskSnapshot.setStatus(status);
        taskSnapshot.setRetryCount(retryTimes);

        for (String server : serverList) {

            Result<Boolean> acknowledgeResult = null;
            try {
                InvocationContext.setRemoteMachine(new RemoteMachine(server, 12 * DEFAULT_INVOKE_TIMEOUT));
                acknowledgeResult = serverService.acknowledge(taskSnapshot);
            } catch (Throwable e) {
                logger.error("[Executor]: acknowledge error"
                        + ", status:" + status
                        + ", retryTimes:" + retryTimes
                        + ", server:" + server
                        + ", instanceId:" + taskSnapshot.getJobInstanceId()
                        + ", id:" + taskSnapshot.getId(), e);
            }

            if (null == acknowledgeResult || !acknowledgeResult.getData().booleanValue()) {
                logger.error("[Executor]: acknowledge failed"
                        + ", status:" + status
                        + ", retryTimes:" + retryTimes
                        + ", server:" + server
                        + ", instanceId:" + taskSnapshot.getJobInstanceId()
                        + ", id:" + taskSnapshot.getId()
                        + ", acknowledgeResult:" + acknowledgeResult);
            } else {
                return;
            }
        }
    }



    public Result<Boolean> acknowledgeRes(TaskSnapshot taskSnapshot, int status, int retryTimes) {
        List<String> serverList = this.clientContext.getClientRemoting().getServerList();
        if (CollectionUtils.isEmpty(serverList)) {
            logger.error("[Executor]: acknowledge serverList is isEmpty error"
                    + ", status:" + status
                    + ", retryTimes:" + retryTimes
                    + ", instanceId:" + taskSnapshot.getJobInstanceId()
                    + ", id:" + taskSnapshot.getId());
            return null;
        }

        /** 随机列表顺序 */
        Collections.shuffle(serverList);

        //如果clientId为空就补上
        if (StringUtil.isBlank(taskSnapshot.getClientId())) {
            taskSnapshot.setClientId(this.clientContext.getClientConfig().getClientId());
        }

        taskSnapshot.setStatus(status);
        taskSnapshot.setRetryCount(retryTimes);

        Result<Boolean> acknowledgeResult = null;

        for (String server : serverList) {
            try {
                InvocationContext.setRemoteMachine(new RemoteMachine(server, 2 * DEFAULT_INVOKE_TIMEOUT));
                acknowledgeResult = serverService.acknowledge(taskSnapshot);
            } catch (Throwable e) {
                logger.error("[Executor]: acknowledge error"
                        + ", status:" + status
                        + ", retryTimes:" + retryTimes
                        + ", server:" + server
                        + ", instanceId:" + taskSnapshot.getJobInstanceId()
                        + ", id:" + taskSnapshot.getId(), e);
            }

            if (null == acknowledgeResult || !acknowledgeResult.getData().booleanValue()) {
                logger.error("[Executor]: acknowledge failed"
                        + ", status:" + status
                        + ", retryTimes:" + retryTimes
                        + ", server:" + server
                        + ", instanceId:" + taskSnapshot.getJobInstanceId()
                        + ", id:" + taskSnapshot.getId()
                        + ", acknowledgeResult:" + acknowledgeResult);
            } else {
                return acknowledgeResult;
            }
        }

        return acknowledgeResult;
    }



//	public void batchAcknowledge(ExecutableTask executableTask, int status, int retryTimes) {
//		List<String> serverList = this.clientContext.getClientRemoting().getServerList();
//		if(CollectionUtils.isEmpty(serverList)) {
//			logger.error("[Executor]: batchAcknowledge serverList is isEmpty error"
//					+ ", status:" + status
//					+ ", retryTimes:" + retryTimes);
//			return ;
//		}
//
//		/** 随机列表顺序 */
//		Collections.shuffle(serverList);
//
//		for(String server : serverList) {
//			InvocationContext.setRemoteMachine(new RemoteMachine(server, 12 * DEFAULT_INVOKE_TIMEOUT));
//			Result<Boolean> batchAcknowledgeResult = serverService.batchAcknowledge(
//					executableTask,
//					status);
//			if(null == batchAcknowledgeResult || ! batchAcknowledgeResult.getData().booleanValue()) {
//				logger.error("[Executor]: batchAcknowledge failed"
//						+ ", status:" + status
//						+ ", retryTimes:" + retryTimes
//						+ ", server:" + server);
//			} else {
//				return ;
//			}
//		}
//	}


    public Result<String> heartBeatCheckJobInstance(int jobType, long jobId, long jobInstanceId) {
        Result<String> result = new Result<String>();

        if (CommonUtil.isSimpleJob(jobType)) {
            if (Constants.ENVIRONMENT_SCX.equals(this.clientContext.getClientConfig().getEnvironment()) && !this.clientContext.getClientConfig().isNewVersion()) {
                result = scxSimplePool.heartBeatCheckJobInstance(jobId, jobInstanceId);
            } else {
                result = simplePool.heartBeatCheckJobInstance(jobId, jobInstanceId);
            }
        } else {

            if (CommonUtil.isLongTimeJob(jobType)) {
                result = longTimePool.heartBeatCheckJobInstance(jobId, jobInstanceId);
            } else if (CommonUtil.isParallelJob(jobType)) {
                result = parallelPool.heartBeatCheckJobInstance(jobId, jobInstanceId);
            }
        }
        return result;
    }

    public Result<Boolean> push(int jobType, long jobId, long jobInstanceId, TaskSnapshot taskSnapshot) {

        if (!CommonUtil.isSimpleJob(jobType)) {

            if (CommonUtil.isLongTimeJob(jobType)) {
                return longTimePool.push(jobId, jobInstanceId, taskSnapshot);
            } else {
                return parallelPool.push(jobId, jobInstanceId, taskSnapshot);
            }

        } else {
            return new Result<Boolean>(false, ResultCode.PUSH_JOB_TYPE_ERROR);
        }

    }

    public Result<Boolean> taskStatesReport(long jobInstanceId, List<Long> taskids) {

        List<String> serverList = this.clientContext.getClientRemoting().getServerList();
        if (CollectionUtils.isEmpty(serverList)) {
            logger.error("[Executor]: taskStatesReport serverList is isEmpty error"
                    + ", instanceId:" + jobInstanceId
                    + ", ids:" + taskids);
            return null;
        }

        /** 随机列表顺序 */
        Collections.shuffle(serverList);

        Result<Boolean> taskStatesReportResult = null;

        for (String server : serverList) {
            try {
                InvocationContext.setRemoteMachine(new RemoteMachine(server, 2 * DEFAULT_INVOKE_TIMEOUT));
                taskStatesReportResult = serverService.longTimeTaskStatesReport(jobInstanceId, taskids);
            } catch (Throwable e) {
                logger.error("[Executor]: taskStatesReport error"
                        + ", server:" + server
                        + ", instanceId:" + jobInstanceId
                        + ", ids:" + taskids, e);
            }

            if (null == taskStatesReportResult || !taskStatesReportResult.getData().booleanValue()) {
                logger.error("[Executor]: taskStatesReport failed"
                        + ", server:" + server
                        + ", instanceId:" + jobInstanceId
                        + ", id:" + taskids
                        + ", taskStatesReportResult:" + taskStatesReportResult);
            } else {
                return taskStatesReportResult;
            }
        }

        return taskStatesReportResult;

    }

    public SimplePool getSimplePool() {
        return simplePool;
    }

    public ScxSimplePool getScxSimplePool() {
        return scxSimplePool;
    }

    public LongTimePool getLongTimePool() {
        return longTimePool;
    }

    public void acknowledgeNode(TaskSnapshot taskSnapshot, int status, int retryTimes) {


        //如果clientId为空就补上
        if (StringUtil.isBlank(taskSnapshot.getClientId())) {
            taskSnapshot.setClientId(this.clientContext.getClientConfig().getClientId());
        }

        taskSnapshot.setStatus(status);
        taskSnapshot.setRetryCount(retryTimes);

        String sendNode = taskSnapshot.getSendNodeAddress();
        taskSnapshot.setSendNodeAddress(null);
        Result<Boolean> acknowledgeResult = null;
        for (int i = 0; i < 3; i++) {
            try {
                InvocationContext.setRemoteMachine(new RemoteMachine(sendNode, 12 * DEFAULT_INVOKE_TIMEOUT));
                acknowledgeResult = nodeServerService.acknowledge(taskSnapshot);
            } catch (Throwable e) {
                logger.error("[Executor]: acknowledge error"
                        + ", status:" + status
                        + ", retryTimes:" + retryTimes
                        + ", sendOld node:" + sendNode
                        + ", instanceId:" + taskSnapshot.getJobInstanceId()
                        + ", id:" + taskSnapshot.getId(), e);
            }

            if (null == acknowledgeResult || !acknowledgeResult.getData().booleanValue()) {
                logger.error("[Executor]: acknowledge failed"
                        + ", status:" + status
                        + ", retryTimes:" + retryTimes
                        + ", sendNode:" + sendNode
                        + ", instanceId:" + taskSnapshot.getJobInstanceId()
                        + ", id:" + taskSnapshot.getId()
                        + ", acknowledgeResult:" + acknowledgeResult);
            } else {
                break;
            }
        }
    }

    public ConcurrentHashMap<String, ConcurrentHashMap<String, ExecutionCounter>> getExecutionCounters(Long jobInstanceId) {
        ConcurrentHashMap<String, ConcurrentHashMap<String, ExecutionCounter>> executionCounterMapByReceiveNode = clientContext.getExecutionCounterTable().get(jobInstanceId);
        if (executionCounterMapByReceiveNode == null) {
            executionCounterMapByReceiveNode = new ConcurrentHashMap<String, ConcurrentHashMap<String, ExecutionCounter>>();
        }
        return executionCounterMapByReceiveNode;
    }


    public void acknowledgeCompensation(TaskSnapshot taskSnapshot, int status, int retryCount) {

    }

    public void acknowledgeCompensationNode(TaskSnapshot taskSnapshot, int status, int retryCount) {

    }

    public boolean isJobInstanceFinished(long jobInstanceId) {

        try {
            BlockingQueue<List<TaskEvent>> queue = clientContext.getGridTaskSender().getTasksForInsertBufferMap().get(jobInstanceId);
            if (queue != null) {
                if (!queue.isEmpty())
                    return false;
            }
            long unFinishedCount = clientContext.getStore().getTaskSnapshotDao().queryUnFinishedTasksCount(jobInstanceId);
            return unFinishedCount == 0;
        } catch (AccessException e) {
            logger.error("isTasksFinished error , jobInstanceId=" + jobInstanceId, e);
            return false;
        }
    }

    public boolean isJobInstanceCanBeDeleted(long jobInstanceId) {
        try {
            long unFinishedCount = clientContext.getStore().getTaskSnapshotDao().queryUnFinishedTasksCountForDelete(jobInstanceId);
            return unFinishedCount == 0;
        } catch (AccessException e) {
            logger.error("isTasksFinished error , jobInstanceId=" + jobInstanceId, e);
            return false;
        }
    }

    public Result<Boolean> doGridJobCleanTask(long jobInstanceId) {
        logger.info("start clean grid job, jobInstanceId=" + jobInstanceId);
        Result<Boolean> result = new Result<Boolean>();
        try {
            clientContext.getCompensationTimer().removeJobInstanceSnapshot(jobInstanceId);
            clientContext.getExecutionCounterTable().remove(jobInstanceId);
            clientContext.getStore().getExecutionCounterDao().deleteByJobInstanceId(jobInstanceId);
            ExecutorService executorService = clientContext.getNodeServerServiceLocal().getExecutorServiceMap().get(jobInstanceId);
            if (executorService != null) {
                executorService.shutdown();
            }
            clientContext.getGridTaskSender().clearInsertBuffer(jobInstanceId);
            clientContext.getNodeServerServiceLocal().getExecutorServiceMap().remove(jobInstanceId);
            clientContext.getGridTaskSender().removeInterruptedJobInstance(jobInstanceId);
            clientContext.getGridTaskSender().getSendManager().clearMergingTaskGroupMap(jobInstanceId);
            result.setData(true);
            result.setResultCode(ResultCode.SUCCESS);
            logger.info("clean grid job finished. jobInstanId=" + jobInstanceId);
        } catch (Throwable throwable) {
            logger.error("failed to do grid job clean task with job instance id " + jobInstanceId, throwable);
            result.setData(false);
            result.setResultCode(ResultCode.FAILURE);
        }
        return result;
    }


    public Result<JobInstanceSnapshot> getJobInstanceById(Long jobInstanceId) {
        List<String> serverList = this.clientContext.getClientRemoting().getServerList();
        if (CollectionUtils.isEmpty(serverList)) {
            logger.error("[Executor]: getJobInstanceById serverList is isEmpty error"
                    + ", instanceId:" + jobInstanceId);
            return null;
        }

        /** 随机列表顺序 */
        Collections.shuffle(serverList);

        Result<JobInstanceSnapshot> jobInstanceResult = null;

        for (String server : serverList) {

            try {
                InvocationContext.setRemoteMachine(new RemoteMachine(server, 6 * DEFAULT_INVOKE_TIMEOUT));
                jobInstanceResult = serverService.getJobInstanceById(jobInstanceId);
            } catch (Throwable e) {
                logger.error("[Executor]: getJobInstanceById error"
                        + ", server:" + server
                        + ", instanceId:" + jobInstanceId);
            }

            if (null == jobInstanceResult || jobInstanceResult.getData()==null) {
                logger.error("[Executor]: getJobInstanceById error"
                        + ", server:" + server
                        + ", instanceId:" + jobInstanceId);
            } else {
                return jobInstanceResult;
            }
        }

        return jobInstanceResult;
    }

    public List<String> getLog(long jobId, long jobInstanceId, String fireTime, int streamType) throws IOException {
        AbstractLogCollector logCollector = LogCollectorFactory.newCollector(jobId, jobInstanceId, fireTime);
        if (streamType == StreamType.STD_OUT.getValue()) {
            //如果读取执行结果，返回正数20行数据
            return logCollector.readLines(0, 20, StreamType.STD_OUT);
        } else {
            //如果读取执行日志，返回倒数20行日志
            return logCollector.tailLines(20, StreamType.STD_ERR);
        }
    }

}
