package com.taobao.eagleeye;

import java.io.IOException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * 在 web.xml 中加入如下配置，可以使 EagleEye 埋点生效
 * <pre>
 * {@code
 * <filter>
 *    <filter-name>EagleEyeFilter</filter-name>
 *    <filter-class>com.taobao.eagleeye.EagleEyeFilter</filter-class>
 *    <init-param>
 *       <param-name>useLocalIp</param-name>
 *       <param-value>true</param-value>
 *    </init-param>
 * </filter>
 *
 * <filter-mapping>
 *    <filter-name>EagleEyeFilter</filter-name>
 *    <url-pattern>/*</url-pattern>
 * </filter-mapping>
 * }
 * </pre>
 * 注意：filter-mapping 的设置，一定要保证它的顺序在其他 mapping 前面，
 * 否则很可能被前面的 Filter 处理请求之后直接返回，导致一直没有埋点。
 * @since 1.5.0-exported
 */
public class EagleEyeFilter implements Filter {

	private static final String USE_LOCAL_IP = "useLocalIp";

	private boolean useLocalIp = false;

	@Override
	public void destroy() {
	}

	@Override
	public void doFilter(ServletRequest request, ServletResponse response,
			FilterChain chain) throws IOException, ServletException {
		if (!(request instanceof HttpServletRequest && response instanceof HttpServletResponse)) {
			chain.doFilter(request, response);
			return;
		}

		HttpServletRequest httpRequest = (HttpServletRequest) request;
		HttpServletResponse httpResponse = (HttpServletResponse) response;
		try {
			String ip = null;
			if (!useLocalIp) {
				ip = EagleEyeRequestTracer.getRemoteAddress(httpRequest);
			}
			String traceId = EagleEyeRequestTracer.getTraceId(httpRequest, ip);
			EagleEyeRequestTracer.startTrace(traceId, httpRequest, httpResponse);
		} catch (Throwable e) {
			// quietly
		}

		try {
			chain.doFilter(request, httpResponse);
		} finally {
			EagleEyeRequestTracer.endTrace(httpRequest, httpResponse);
		}
	}

	@Override
	public void init(FilterConfig filterConfig) throws ServletException {
		String uselocal = filterConfig.getInitParameter(USE_LOCAL_IP);
		if(uselocal != null && "true".equals(uselocal)){
			useLocalIp = true;
		}
	}
}
